<?php

class ControllerAndroidServicePartnerSaleOrder extends Controller
{
    private $json;
    private $error;
    private $model_name;

    public function index()
    {
        //Load Model
        $this->load->model('android/service_partner/sale_order');
        $this->model_name = $this->model_android_service_partner_sale_order;

        //Load required file
        require DIR_SYSTEM . "/library/start_api.php";
    }

    #region api call function
    protected function get_list()
    {
        $filters = $this->json;
        $search = !empty($filters['search']) ? $filters['search'] : '';
        $page = !empty($filters['page']) ? $filters['page'] : 1;
        $status = !empty($filters['status']) ? $filters['status'] : '';
        $limit = !empty($filters['limit']) ? $filters['limit'] : $this->config->get('config_limit_admin');

        $filter_data = [
            'search' => $search,
            'status' => $status,
            'start'  => ($page - 1) * $limit,
            'limit'  => (int)$limit
        ];

        $sale_orders = $this->model_name->getSaleOrders($filter_data);

        $result = $this->formattedOrderList($sale_orders);

        if (!empty($result)) {
            $output = $this->response->setCachedResponse(200, $result, ' Success', "service_partner");
            $this->api->trackResponse($output);
            return;
        } else {
            $output = $this->response->setCachedResponse(204, array(), " No result!", "service_partner");
            $this->api->trackResponse($output);
            return;
        }
    }

    protected function get_detail()
    {
        $sale_order_id = !empty($this->request->get['sale_order_id']) ? $this->request->get['sale_order_id'] : 0;

        if (!empty($sale_order_id)) {
            $so_info = $this->model_name->getSaleOrder($sale_order_id);

            if (!empty($so_info)) {
                $total_data = !empty($so_info['service_partner_total_json']) ?  $so_info['service_partner_total_json'] : [];
                $totals = $this->formattedTotal($total_data);

                $prescription_data = !empty($so_info['service_partner_prescription_json']) ?  $so_info['service_partner_prescription_json'] : [];
                $prescriptions = $this->formattedPrescriptions($prescription_data);

                $image_data = !empty($so_info['image_json']) ? $so_info['image_json'] : [];
                $images = json_decode($image_data, true);

                $action = $this->getOrderAction($so_info);

                $order_summary = $this->getOrderSummary($so_info);

                //get order detail from etcm
                $order_detail = [];
                $order_history =  [];
                $path_order = 'etcm/service_partner/service_partner/get_order_detail';
                $detail_result = $this->connectCurl($path_order, array("purchase_order_id" => $so_info['etcm_purchase_order_id']));
                if (!empty($detail_result)) {
                    if (empty($detail_result['error'])) {
                        $order_detail = $detail_result['response'];

                        //order_history_part
                        foreach ($order_detail['history'] as $history) {
                            $order_history[] = array(
                                'date' => date_format(date_create($history['date_added']), "j M H:i"),
                                'text' => $history['comment']
                            );
                        }
                    }
                }

                //get cancel reason from etcm
                /*
                    // $data = ['domain' => HTTP_SERVER];
                    // $path = 'etcm/service_partner/service_partner/get_cancel_reason';
                    // $reasons_result = $this->connectCurl($path, $data);
                    // $reasons = [];
                    // if (!empty($reasons_result)) {
                    //     if (empty($reasons_result['error'])) {
                    //         $reasons = $reasons_result['response'];
                    //     }
                    // }
                */
                $reasons = null;
                $carriers = $this->model_name->getCarriers();

                $shipping_address = $so_info['shipping_type'] == 'PICKUP_FROM_STORE' ? 'Pickup from store' : $so_info['shipping_address'];

                $shipping_type = str_replace('_', ' ', ucfirst(strtolower($so_info['shipping_type'])));


                if ($so_info['etcm_status_code'] == 'TO_PAY') {
                    $payment_text = 'UNPAID';
                } else {
                    if ($order_detail['payment_method'] == 'free_checkout') {
                        $payment_text = 'ETCM Pay';
                    } elseif ($order_detail['payment_method'] == 'kiple') {
                        $payment_title = !empty($order_detail['payment_title']) ? "(" . $order_detail['payment_title'] . ")" : "";
                        $payment_text = "Online $payment_title";
                    } else {
                        $payment_text = $order_detail['payment_method'];
                    }
                }

                //estimate text
                $estimated_text = $so_info['estimated_date'];
                if (date('Y', strtotime($estimated_text)) != '1970') {
                    $est_date = date('j M', strtotime($estimated_text));
                    $est_time = date('h:i A', strtotime($estimated_text));

                    if ($so_info['etcm_status_code'] == 'TO_SHIP' && !empty($so_info['date_prepared'])) {
                        $estimated_text_en = "Estimate dispensing process will complete on $est_date at $est_time";
                        $estimated_text_cn = "配药流程预计将完成于 $est_date $est_time";
                        if ($so_info['shipping_type'] == 'DELIVERY') {
                            $estimated_text_en = "Order will ship out by $est_date at $est_time";
                            $estimated_text_cn = "订单将发货于 $est_date $est_time";
                        }
                    }
                    if ($so_info['etcm_status_code'] == 'TO_RECEIVE') {
                        $estimated_text_en = "";
                        $estimated_text_cn = "";
                        if ($so_info['shipping_type'] != 'DELIVERY') {
                            $estimated_text_en = "Ready to collect by $est_date after $est_time";
                            $estimated_text_cn = "将被提取于 $est_date $est_time 之后";
                        }
                    }
                } elseif ($so_info['etcm_status_code'] == 'COMPLETED') {
                    $estimated_text_en = "Order Received";
                    $estimated_text_cn = "订单已完成";
                } elseif ($so_info['etcm_status_code'] == 'CANCELLED') {
                    $estimated_text_en = "Order Cancelled";
                    $estimated_text_cn = "订单已取消";
                } else {
                    $estimated_text_en = "Estimated: $estimated_text";
                    $estimated_text_cn = "预计" . str_replace(' working day(s)', "天的工作日", $estimated_text);
                    // $estimated_text_cn = "预计1-2天的工作日";
                }

                $estimated_text = $this->session->data['language_id'] == 1 ? $estimated_text_en : $estimated_text_cn;

                $pickup_session = !empty($this->config->get('config_spp_pickup_session')) ? $this->config->get('config_spp_pickup_session') : [];
                $pickup_initial_time = null;
                foreach ($pickup_session as $key => $time) {
                    if ($time > date('H:i:s') && empty($pickup_initial_time)) {
                        $pickup_initial_time = date('h:i A', strtotime($time));
                    }
                    $pickup_session[$key] = date('h:i A', strtotime($time));
                }

                $tracking_no = (!empty($so_info['tracking_no']) && $so_info['shipping_type'] == 'DELIVERY') ? trim($so_info['tracking_no']) : null;
                $track_word = explode(' ', $tracking_no);
                $tracking_no = array_pop($track_word);

                $tracking_link = '';
                $tracking_no = trim($tracking_no);
                if (strtolower(substr($tracking_no, 0, 3)) === 'znd') {
                    $tracking_link = "https://tracking.my/lineclear/" . $tracking_no;
                } elseif (!empty($tracking_no)) {
                    $tracking_link = "https://www.aftership.com/track/" . $tracking_no;
                } else {
                    $tracking_link = null;
                }

                $return_data = [
                    'customer' => [
                        'customer_id' => (int)$so_info['customer_id'],
                        'customer_name' => $so_info['customer_name'],
                        'customer_telephone' => $so_info['customer_telephone'],
                        'customer_email' => $so_info['customer_email'],
                    ],
                    'delivery_address' => [
                        'shipping_type' => $shipping_type,
                        'estimated_date' => $estimated_text,
                        'shipping_address' => $shipping_address,
                    ],
                    'service_partner' => [
                        'service_partner_id' => (int)$so_info['service_partner_id'],
                        'service_partner_name' => $so_info['service_partner_name'],
                        'service_type' => $this->getServiceTypeName($so_info['service_type']),
                    ],
                    'medical' => [
                        'medical_record_id' => $so_info['medical_record_id'],
                        'prescriptions' => $prescriptions,
                        'images' => $images,
                        'shipping' => (string)$totals['total_data']['shipping'],
                        'shipping_text' => $totals['total_data']['shipping_text'],
                        'total' => $totals['total_data']['total'],
                        'total_text' => $totals['total_data']['total_text'],
                    ],
                    'payment' => [
                        'payment_method' => $payment_text,
                        'payment_address' => $so_info['payment_address']
                    ],
                    'total' => $totals['totals'],
                    'order' => [
                        'sale_order_id' => (int)$so_info['sale_order_id'],
                        'doc_no' => $so_info['doc_no'],
                        'doc_date' => date('d M Y H:i:s', strtotime($so_info['doc_date'])),
                        'status' => $so_info['etcm_status_code'],
                        'shipping_type' => $so_info['shipping_type'],
                        'estimated_date' => $estimated_text,
                        'date_ordered' => !empty($so_info['date_ordered']) ? date('d M Y H:i:s', strtotime($so_info['date_ordered'])) : null,
                        'date_payment' => !empty($so_info['date_payment']) ? date('d M Y H:i:s', strtotime($so_info['date_payment'])) : null,
                        'date_shipped' => !empty($so_info['date_shipped']) ? date('d M Y H:i:s', strtotime($so_info['date_shipped'])) : null,
                        'date_completed' => !empty($so_info['date_completed']) ? date('d M Y H:i:s', strtotime($so_info['date_completed'])) : null,
                        'date_cancelled' => !empty($so_info['date_cancelled']) ? date('d M Y H:i:s', strtotime($so_info['date_cancelled'])) : null,
                        'tracking_no' => (!empty($so_info['tracking_no']) && $so_info['shipping_type'] == 'DELIVERY') ? trim($so_info['tracking_no']) : null,
                        'tracking_no_link' => $tracking_link,
                        'proof_of_delivery' => !empty($so_info['proof_of_delivery']) ? $so_info['proof_of_delivery'] : null
                    ],
                    'order_summary' => $order_summary,
                    'order_history' => $order_history,
                    'action' => $action,
                    'prepare_initial_date' => date("Y-m-d", strtotime('+3 hours')),
                    'prepare_initial_time' => date("h:i A", strtotime('+3 hours')),
                    'prepare_time_suggestion' => [date("h:i A", strtotime('+3 hours'))],
                    'pickup_initial_date' => date("Y-m-d"),
                    'pickup_initial_time' => $pickup_initial_time,
                    'pickup_time_suggestion' => $pickup_session,
                    'cancel_reason' => $reasons,
                    'carriers' => $carriers,
                    'cancelled_reason' => !empty($so_info['cancel_reason']) ? $so_info['cancel_reason'] : null
                ];
                $output = $this->response->setCachedResponse(200, $return_data, 'Success', "service_partner");
                $this->api->trackResponse($output);
                return;
            } else {
                $output = $this->response->setCachedResponse(400, array(), "Sale order id is invalid", "service_partner");
                $this->api->trackResponse($output);
                return;
            }
        } else {
            $output = $this->response->setCachedResponse(400, array(), "Sale order id is required", "service_partner");
            $this->api->trackResponse($output);
            return;
        }
    }

    protected function get_sale_order_statuses()
    {
        $canAccess = $this->load->controller('android/common/permission/canAccess', 'service_partner/sale_order');
        if ($canAccess) {
            $result = $this->getSaleOrderStatuses();
        } else {
            $result = array();
        }

        if (!empty($result)) {

            $output = $this->response->setCachedResponse(200, $result, 'Success', "service_partner");
            return;
        } else {
            $output = $this->response->setCachedResponse(204, array(), " No result", "service_partner");
            $this->api->trackResponse($output);
            return;
        }
    }

    protected function cancel_order()
    {
        $sale_order_id = !empty($this->json['sale_order_id']) ? $this->json['sale_order_id'] : 0;
        $cancel_reason = !empty($this->json['cancel_reason']) ? trim($this->json['cancel_reason']) : '';
        if (!empty($sale_order_id) && !empty($cancel_reason)) {
            $so_info = $this->model_name->getSaleOrder($sale_order_id);
            $etcm_purchase_order_id = $so_info['etcm_purchase_order_id'];
            $update_data = array(
                "status_id" => 7,
                "etcm_status_code" => "CANCELLED",
                "date_cancelled" => date('Y-m-d H:i:s'),
                "cancel_reason" => $cancel_reason,
                "etcm_purchase_order_id" => $etcm_purchase_order_id
            );

            $this->model_name->updateSO($update_data);

            $post_data = array(
                // "tracking_no"   => $tracking_no,
                "status_code"   => $update_data['etcm_status_code'],
                // "date_payment"  => $date_payment,
                // "date_shipped"  => $date_shipped,
                // "date_rated"    => $date_rated,
                // "rate_details"  => $rate_details,
                // "date_completed"    => $date_completed,
                "date_cancelled"    => $update_data['date_cancelled'],
                "cancel_reason" => $cancel_reason,
                "purchase_order_id" => $etcm_purchase_order_id,
                "from" => "MERCHANT"
            );

            $path = 'etcm/service_partner/service_partner/update_po';
            $result = $this->connectCurl($path, $post_data);
            if (!empty($result)) {
                if (!empty($result['error'])) {
                    $output = $this->response->setCachedResponse(204, new stdClass(), $result['message'], "service_partner");
                    return;
                } else {
                    $output = $this->response->setCachedResponse(200, new stdClass(), $result['message'], "service_partner");
                    return;
                }
            } else {
                $output = $this->response->setCachedResponse(204, new stdClass(), "API Connection Errors : Please check the internet connection or API connection!", "service_partner");
                $this->api->trackResponse($output);
                return;
            }
        } else {
            $output = $this->response->setCachedResponse(400, new stdClass(), "sale order id & cancel reason required", "service_partner");
            $this->api->trackResponse($output);
            return;
        }
    }

    protected function ship_order()
    {
        $sale_order_id = !empty($this->json['sale_order_id']) ? $this->json['sale_order_id'] : 0;
        $tracking_no = !empty($this->json['tracking_no']) ? $this->json['tracking_no'] : '';
        $carrier_id = !empty($this->json['carrier_id']) ? $this->json['carrier_id'] : 0;
        if (!empty($sale_order_id) && !empty($tracking_no)) {
            $so_info = $this->model_name->getSaleOrder($sale_order_id);
            $etcm_purchase_order_id = $so_info['etcm_purchase_order_id'];
            $update_data = array(
                "status_id" => 3,
                "etcm_status_code" => "TO_RECEIVE",
                "date_shipped" => date('Y-m-d H:i:s'),
                "carrier_id" => $carrier_id,
                "tracking_no" => $tracking_no,
                "etcm_purchase_order_id" => $etcm_purchase_order_id
            );

            $this->model_name->updateSO($update_data);

            $post_data = array(
                "tracking_no"   => $tracking_no,
                "status_code"   => $update_data['etcm_status_code'],
                // "date_payment"  => $date_payment,
                "date_shipped"  => $update_data['date_shipped'],
                // "date_rated"    => $date_rated,
                // "rate_details"  => $rate_details,
                // "date_completed"    => $date_completed,
                // "date_cancelled"    => $update_data['date_cancelled'],
                // "cancel_reason" => $cancel_reason,
                "purchase_order_id" => $etcm_purchase_order_id,
                "from" => "MERCHANT"
            );

            $path = 'etcm/service_partner/service_partner/update_po';
            $result = $this->connectCurl($path, $post_data);
            if (!empty($result)) {
                if (!empty($result['error'])) {
                    $output = $this->response->setCachedResponse(204, new stdClass(), $result['message'], "service_partner");
                    return;
                } else {
                    $output = $this->response->setCachedResponse(200, new stdClass(), $result['message'], "service_partner");
                    return;
                }
            } else {
                $output = $this->response->setCachedResponse(204, new stdClass(), "API Connection Errors : Please check the internet connection or API connection!", "service_partner");
                $this->api->trackResponse($output);
                return;
            }
        } else {
            $output = $this->response->setCachedResponse(400, new stdClass(), "sale order id & tracking no required", "service_partner");
            $this->api->trackResponse($output);
            return;
        }
    }

    protected function pick_order()
    {
        $sale_order_id = !empty($this->json['sale_order_id']) ? $this->json['sale_order_id'] : 0;
        $estimate_date = !empty($this->json['estimate_date']) ? $this->json['estimate_date'] : '';
        $estimate_time = !empty($this->json['estimate_time']) ? $this->json['estimate_time'] : '';

        if (!empty($sale_order_id) && !empty($estimate_date) && !empty($estimate_time)) {
            $so_info = $this->model_name->getSaleOrder($sale_order_id);
            $etcm_purchase_order_id = $so_info['etcm_purchase_order_id'];
            $update_data = array(
                "status_id" => 3,
                "etcm_status_code" => "TO_RECEIVE",
                "date_shipped" => date('Y-m-d H:i:s'),
                "estimated_date" => "$estimate_date $estimate_time",
                "etcm_purchase_order_id" => $etcm_purchase_order_id
            );

            $this->model_name->updateSO($update_data);

            $post_data = array(
                "status_code"   => $update_data['etcm_status_code'],
                "date_shipped"  => $update_data['date_shipped'],
                "estimated_date"    => $update_data['estimated_date'],
                "purchase_order_id" => $etcm_purchase_order_id,
                "from" => "MERCHANT"
            );

            $path = 'etcm/service_partner/service_partner/update_po';
            $result = $this->connectCurl($path, $post_data);
            if (!empty($result)) {
                if (!empty($result['error'])) {
                    $output = $this->response->setCachedResponse(204, new stdClass(), $result['message'], "service_partner");
                    return;
                } else {
                    $output = $this->response->setCachedResponse(200, new stdClass(), $result['message'], "service_partner");
                    return;
                }
            } else {
                $output = $this->response->setCachedResponse(204, new stdClass(), "API Connection Errors : Please check the internet connection or API connection!", "service_partner");
                $this->api->trackResponse($output);
                return;
            }
        } else {
            $output = $this->response->setCachedResponse(400, new stdClass(), "sale order id & estimate date time required", "service_partner");
            $this->api->trackResponse($output);
            return;
        }
    }

    protected function prepare_order()
    {
        $sale_order_id = !empty($this->json['sale_order_id']) ? $this->json['sale_order_id'] : 0;
        $estimate_date = !empty($this->json['estimate_date']) ? $this->json['estimate_date'] : '';
        $estimate_time = !empty($this->json['estimate_time']) ? $this->json['estimate_time'] : '';
        if (!empty($sale_order_id)) {
            $so_info = $this->model_name->getSaleOrder($sale_order_id);
            $etcm_purchase_order_id = $so_info['etcm_purchase_order_id'];
            $update_data = array(
                "etcm_status_code" => "TO_SHIP",
                "date_prepared" => date('Y-m-d H:i:s'),
                "estimated_date" => "$estimate_date $estimate_time",
                "etcm_purchase_order_id" => $etcm_purchase_order_id
            );

            $this->model_name->updateSO($update_data);

            $post_data = array(
                "status_code"       => $update_data['etcm_status_code'],
                "date_prepared"     => $update_data['date_prepared'],
                "estimated_date"    => $update_data['estimated_date'],
                "purchase_order_id" => $etcm_purchase_order_id,
                "from"              => "MERCHANT"
            );

            $path = "etcm/service_partner/service_partner/update_po";
            $result = $this->connectCurl($path, $post_data);
            if (!empty($result)) {
                if (!empty($result['error'])) {
                    $output = $this->response->setCachedResponse(204, new stdClass(), $result['message'], "service_partner");
                    return;
                } else {
                    $output = $this->response->setCachedResponse(200, new stdClass(), $result['message'], "service_partner");
                    return;
                }
            } else {
                $output = $this->response->setCachedResponse(204, new stdClass(), "API Connection Errors : Please check the internet connection or API connection!", "service_partner");
                $this->api->trackResponse($output);
                return;
            }
        } else {
            $output = $this->response->setCachedResponse(400, new stdClass(), "sale order id required", "service_partner");
            $this->api->trackResponse($output);
            return;
        }
    }

    protected function complete_order()
    {
        $sale_order_id = !empty($this->json['sale_order_id']) ? $this->json['sale_order_id'] : 0;
        $image = !empty($this->json['image']) ? $this->json['image'] : '';

        if (!empty($sale_order_id) && !empty($image)) {
            $so_info = $this->model_name->getSaleOrder($sale_order_id);
            if (!empty($so_info['etcm_purchase_order_id'])) {
                $post_data = array(
                    "purchase_order_id" => $so_info['etcm_purchase_order_id'],
                    "image" => $image
                );
                $path = 'etcm/service_partner/service_partner/merchant_complete_order';
                $result = $this->connectCurl($path, $post_data);
                if (!empty($result)) {
                    if (!empty($result['error'])) {
                        $output = $this->response->setCachedResponse(204, new stdClass(), $result['message'], "service_partner");
                        return;
                    } else {
                        //save img_url
                        $img_url = !empty($result['response']['img_url']) ? $result['response']['img_url'] : '';
                        $date_completed = !empty($result['response']['date_completed']) ? $result['response']['date_completed'] : '';

                        $update_data = [
                            "status_id"         => 5,
                            "etcm_status_code"  => 'COMPLETED',
                            "date_completed"    => $date_completed,
                            "proof_of_delivery" => $img_url,
                            "etcm_purchase_order_id" => $so_info['etcm_purchase_order_id']
                        ];

                        $this->model_name->updateSO($update_data);

                        $output = $this->response->setCachedResponse(200, new stdClass(), $result['message'], "service_partner");
                        return;
                    }
                } else {
                    $output = $this->response->setCachedResponse(204, new stdClass(), "API Connection Errors : Please check the internet connection or API connection!", "service_partner");
                    $this->api->trackResponse($output);
                    return;
                }
            } else {
                $output = $this->response->setCachedResponse(400, new stdClass(), "sale order id not valid", "service_partner");
                $this->api->trackResponse($output);
                return;
            }
        } else {
            $output = $this->response->setCachedResponse(400, new stdClass(), "sale order id & image required", "service_partner");
            $this->api->trackResponse($output);
            return;
        }
    }

    #endregion

    #region callback function for etcm server
    public function add_sale_order()
    {
        $sale_order_id = 0;
        if (!empty($this->json)) {
            $sale_order_id = $this->model_name->saveSO($this->json);
        }

        $so_info = $this->model_name->getSaleOrder($sale_order_id);
        $this->sendNotification($so_info['etcm_purchase_order_id']);

        $output = $this->response->setCachedResponse(200, array("sale_order_id" => $sale_order_id), 'success', "service_partner");
        $this->api->trackResponse($output);
        return;
    }

    public function update_so()
    {
        $data = $this->json;
        $tracking_no = !empty($data['tracking_no']) ? $data['tracking_no'] : '';
        $status_id = !empty($data['status_id']) ? $data['status_id'] : '';
        $etcm_status_code = !empty($data['etcm_status_code']) ? $data['etcm_status_code'] : '';
        $date_payment = !empty($data['date_payment']) ? $data['date_payment'] : '';
        $date_prepared = !empty($data['date_prepared']) ? $data['date_prepared'] : '';
        $date_shipped = !empty($data['date_shipped']) ? $data['date_shipped'] : '';
        $date_rated = !empty($data['date_rated']) ? $data['date_rated'] : '';
        $rate_details = !empty($data['rate_details']) ? $data['rate_details'] : '';
        $date_completed = !empty($data['date_completed']) ? $data['date_completed'] : '';
        $date_cancelled = !empty($data['date_cancelled']) ? $data['date_cancelled'] : '';
        $cancel_reason = !empty($data['cancel_reason']) ? $data['cancel_reason'] : '';
        $etcm_purchase_order_id = !empty($data['etcm_purchase_order_id']) ? $data['etcm_purchase_order_id'] : '';
        $update_data = array(
            "tracking_no"       => $tracking_no,
            "status_id"         => $status_id,
            "etcm_status_code"  => $etcm_status_code,
            "date_payment"      => $date_payment,
            "date_prepared"     => $date_prepared,
            "date_shipped"      => $date_shipped,
            "date_rated"        => $date_rated,
            "rate_details"      => $rate_details,
            "date_completed"    => $date_completed,
            "date_cancelled"    => $date_cancelled,
            "cancel_reason"     => $cancel_reason,
            "etcm_purchase_order_id"    => $etcm_purchase_order_id
        );
        $this->model_name->updateSO($update_data);

        $this->sendNotification($etcm_purchase_order_id);
    }

    public function update_shipping()
    {
        $data = $this->json;
        $update_data = array(
            "etcm_purchase_order_id"    => $data['etcm_purchase_order_id'],
            "shipping_address"          => $data['shipping_address'],
            "shipping_address_json"     => $data['shipping_address_json'],
        );
        $this->model_name->updateShipping($update_data);
    }

    public function get_price_stock()
    {
        $data = $this->json;
        $product_ids = $data['product_ids'];
        $result = $this->model_name->getPriceStockByIds($product_ids);
        $output = $this->response->setCachedResponse(200, array("result" => $result), 'success', "service_partner");
        $this->api->trackResponse($output);
        return;
    }

    #endregion function

    #region private function

    private function formattedOrderList($orders)
    {
        $data = [];

        $statuses = $this->getSaleOrderStatuses();
        $arr_status = array();
        foreach ($statuses as $status) {
            $arr_status[$status['code']] = [
                "id"        => $status['id'],
                "status"    => $status['status'],
                "code"      => $status['code'],
                "name"      => $status['name'],
                "color"     => $status['color'],
                "sort_order" => $status['sort_order'],
            ];
        }

        if (!empty($orders)) {
            foreach ($orders as $so) {

                $estimated_text = $so['estimated_date'];
                if (date('Y', strtotime($estimated_text)) == '1970') {
                    $estimated_text = null;
                }

                switch ($so['etcm_status_code']) {
                    case 'TO_PAY':
                        $status_text_en = "Pending payment";
                        $status_text_cn = "待付款";
                        break;
                    case  'TO_SHIP':
                        if (!empty($so['date_prepared'])) {
                            $date_text = date_format(date_create($so['date_prepared']), "j M Y");
                            $status_text_en = 'Started dispense on ' . $date_text;
                            $status_text_cn = '在' . $date_text  . '开始配药';
                        } else {
                            $date_text = date_format(date_create($so['date_ordered']), "j M Y");
                            $status_text_en = 'Order placed on ' . $date_text;
                            $status_text_cn = '订单已付款 ' . $date_text;
                        }
                        break;
                    case 'TO_RECEIVE':
                        if ($so['shipping_type'] == 'DELIVERY') {
                            $date_text = date_format(date_create($so['date_shipped']), "j M Y");
                            $status_text_en = 'Shipped on ' . $date_text;
                            $status_text_cn = '订单运送中 ' . $date_text;
                        } else {
                            $collect_date = !empty($estimated_text) ? $estimated_text : $so['date_shipped'];
                            $date_text = date_format(date_create($collect_date), "j M Y");
                            $status_text_en = 'Collect by ' . $date_text;
                            $status_text_cn = '已提取 ' . $date_text;
                        }
                        break;
                    case 'COMPLETED':
                        $date_text =  date_format(date_create($so['date_completed']), "j M Y");
                        $status_text_en = "Completed on $date_text";
                        $status_text_cn = "完成于 $date_text";
                        break;
                    case 'CANCELLED':
                        $date_text =  date_format(date_create($so['date_cancelled']), "j M Y");
                        $status_text_en = "Cancelled on $date_text";
                        $status_text_cn = "取消日期 $date_text";
                        break;
                    default:
                        $status_text_en = '';
                        $status_text_cn = '';
                }

                if (!empty($so['etcm_status_code'])) {
                    if ($so['etcm_status_code'] == 'TO_SHIP' && empty($so['date_prepared'])) {
                        $so['etcm_status_code'] = 'TO_PAY';
                    }
                    $status_name = str_replace('_', ' ', $so['etcm_status_code']);
                } else {
                    $status_name = null;
                }

                if (!empty($arr_status[$so['etcm_status_code']]['name'])) {
                    $status_name = $arr_status[$so['etcm_status_code']]['name'];
                    if ($so['etcm_status_code'] == 'TO_RECEIVE') {
                        if ($so['shipping_type'] == 'DELIVERY') {
                            $status_name = $this->session->data['language_id'] == 1 ? 'Shipped' : '已出货';
                        } else {
                            $status_name = $this->session->data['language_id'] == 1 ? 'Self Collect' : '待自取';
                        }
                    }
                }

                $status_text = $this->session->data['language_id'] == 1 ? $status_text_en : $status_text_cn;

                $data[] = [
                    'sale_order_id' => (int)$so['sale_order_id'],
                    'doc_no' => $so['doc_no'],
                    'doc_date' => date('d M Y', strtotime($so['doc_date'])),
                    'status_text' => (string)$status_text,
                    'status' => $so['etcm_status_code'],
                    'status_name' => $status_name,
                    'color' => !empty($arr_status[$so['etcm_status_code']]['color']) ? $arr_status[$so['etcm_status_code']]['color'] : '#eeeeee',
                    'customer_id' => (int)$so['customer_id'],
                    'customer_name' => $so['customer_name'],
                    'customer_telephone' => $so['customer_telephone'],
                    'customer_email' => $so['customer_email'],
                    'service_partner_name' => $so['service_partner_name'],
                    'service_type' => $so['service_type'],
                    'shipping_type' => $so['shipping_type'],
                    'total' => (float)$so['service_partner_total'],
                    'total_text' => $so['service_partner_total_text'],
                ];
            }
        }

        return $data;
    }

    private function formattedTotal($totals)
    {
        $data = [];
        $total_amount = 0;
        $total_text = '';
        $shipping = 0;
        $shipping_text = '';
        $data['totals'] = [];
        $data['total_data'] = [];
        if (!empty($totals)) {
            $totals = json_decode($totals, true);
            foreach ($totals as $total) {

                //translate title
                switch ($total['code']) {
                    case 'sub_total':
                        $title_en = "Subtotal";
                        $title_cn = "商品小计";
                        break;
                    case 'tax':
                        $title_en = "Tax";
                        $title_cn = "税额";
                        break;
                    case 'rounding':
                        $title_en = "Rounding";
                        $title_cn = "舍入";
                        break;
                    case 'shipping':
                        $title_en = "Shipping";
                        $title_cn = "运输费";
                        break;
                    case 'credit':
                        $title_en = "ETCM Pay";
                        $title_cn = "ETCM 支付";
                        break;
                    case 'total':
                        $title_en = "Total";
                        $title_cn = "付款总额";
                        break;
                    default:
                        $title_en = $total['title'];
                        $title_cn = $total['title'];
                }

                $title = $this->session->data['language_id'] == 1 ? $title_en : $title_cn;

                $data['totals'][] = [
                    'title' => $title,
                    'value' => number_format($total['value'], 2),
                    'text' => !empty($total['text']) ? $total['text'] : $this->currency->format($total['value'], $this->config->get('config_currency'))
                ];

                if ($total['code'] == 'shipping') {
                    $shipping = number_format($total['value'], 2);
                    $shipping_text = !empty($total['text']) ? $total['text'] : $this->currency->format($total['value'], $this->config->get('config_currency'));
                }

                if ($total['code'] == 'sub_total') {
                    $total_amount = number_format($total['value'], 2);
                    $total_text = !empty($total['text']) ? $total['text'] : $this->currency->format($total['value'], $this->config->get('config_currency'));
                }
            }

            $data['total_data'] = [
                'shipping' => $shipping,
                'shipping_text' => $shipping_text,
                'total' => $total_amount,
                'total_text' => $total_text
            ];
        }
        return $data;
    }

    private function convertPresName($name)
    {
        if ($this->session->data['language_id'] == 1) {
            $name = str_replace('处方', 'Prescription', $name);
        } else {
            $name = str_replace('Prescription', '处方', $name);
        }
        return $name;
    }

    private function formattedPrescriptions($prescriptions)
    {
        $data = [];
        if (!empty($prescriptions)) {
            $prescriptions = json_decode($prescriptions, true);
            foreach ($prescriptions as $key => $prescription) {
                $data[$key] = [
                    'name' => (string)$this->convertPresName($prescription['name']),
                    'times' => '共' . $prescription['times'] . '剂',
                    'day' => '每日' . $prescription['day'] . '次',
                    'consume_method' => !empty($prescription['consume_method']) ? $prescription['consume_method'] : '',
                    'take_type' => !empty($prescription['take_type']) ? $prescription['take_type'] : '',
                    "medical_instruction" => !empty($prescription['medical_instruction']) ? $prescription['medical_instruction'] : '',
                    // 'total_products' => $prescription['total_products'] . ' 药品',
                    // 'total_weights' => $prescription['total_weights'] . $prescription['uom'],
                    // 'total_amount' => $prescription['total_amount_text'],
                    'products' => [],
                ];

                if (!empty($prescription['products'])) {
                    foreach ($prescription['products'] as $product) {
                        $product_name = $this->getProductName($product['product_id']);
                        $product_name = !empty($product_name) ? $product_name : $product['customer_product_name'];
                        $data[$key]['products'][] = [
                            // 'product_id' => (int)$product['customer_product_id'],
                            'product_name' => $product_name,
                            'quantity' => number_format($product['quantity'], 2),
                            // 'times' => (int)$product['times'],
                            // 'total_weight' => number_format($product['total_weight'], 2) . $product['uom'],
                            'total_weight' => number_format($product['quantity'], 2) . $product['uom'],
                            // 'total' => $product['total_text'],
                            'total' =>  $this->currency->format($product['price'], $this->config->get('config_currency')),
                        ];
                    }
                }
            }
        }
        return $data;
    }

    private function getProductName($etcm_product_id)
    {
        $product_info = $this->model_name->getProductDetailByEtcmProductId($etcm_product_id);
        $product_name = '';
        if (!empty($product_info['name'])) {
            $product_name = $product_info['name'];
        }
        return $product_name;
    }

    private function getOrderAction($order)
    {
        $default_greeting = "";
        $default_greeting = urlencode($default_greeting);
        $status = $order['etcm_status_code'];
        $customer_telephone = $this->formatTelephone($order['customer_telephone']);
        $data = [];
        $contact_store = array(
            'action_type' => 'link',
            'action' => "https://api.whatsapp.com/send?phone=$customer_telephone&text=" . $default_greeting,
            'status' => true,

            // 'code' =>null,
            // 'type' => null,
            // 'float_bottom' => null,
        );


        if ($status != 'COMPLETED' && $status != 'CANCELLED' && $status != 'TO_RECEIVE') {
            $cancel_button = array(
                'action_type' => 'api',
                'action' => null,
                'status' => true,
            );
        } else {
            $cancel_button = array(
                'action_type' => 'api',
                'action' => null,
                'status' => false,
            );
        }

        if ($status == 'TO_SHIP' && !empty($order['date_prepared']) && $order['shipping_type'] == 'DELIVERY') {
            // $cancel_button['status'] = false;

            $ship_button = array(
                'action_type' => 'api',
                'action' => null,
                'status' => true,
            );
        } else {
            $ship_button = array(
                'action_type' => 'api',
                'action' => null,
                'status' => false,
            );
        }

        if ($status == 'TO_SHIP' && empty($order['date_prepared'])) {
            $prepare_button = array(
                'action_type' => 'api',
                'action' => null,
                'status' => true,
            );
        } else {
            $prepare_button = array(
                'action_type' => 'api',
                'action' => null,
                'status' => false,
            );
        }

        if ($status == 'TO_SHIP' && !empty($order['date_prepared']) && $order['shipping_type'] != 'DELIVERY') {
            // $cancel_button['status'] = false;

            $pickup_button = array(
                'action_type' => 'api',
                'action' => null,
                'status' => true,
            );
        } else {
            $pickup_button = array(
                'action_type' => 'api',
                'action' => null,
                'status' => false,
            );
        }

        if ($status == 'TO_RECEIVE' && $order['shipping_type'] == 'PICKUP_FROM_STORE') {
            $received_button = array(
                'action_type' => 'api',
                'action' => null,
                'status' => true,
            );
        } else {
            $received_button = array(
                'action_type' => 'api',
                'action' => null,
                'status' => false,
            );
        }

        $data = [
            'contact_store' => $contact_store,
            'pay_button' => null,
            'prepare_button' => $prepare_button,
            'ship_button' => $ship_button,
            'pickup_button' => $pickup_button,
            'received_button' => $received_button,
            'cancel_button' => $cancel_button,
            'rate_button' => null,
        ];
        return $data;
    }

    private function getSaleOrderStatuses()
    {
        $data = [];
        $language = !empty($_SERVER['HTTP_ACCEPT_LANGUAGE']) ? $_SERVER['HTTP_ACCEPT_LANGUAGE'] : 'cn';
        if ($language == 'en') {
            $data = [
                array(
                    'id'   => 1,
                    'status' => 0,
                    'code' => 'ALL',
                    'name' => 'All',
                    'image' => "https://www.etcm.me/image/doctorapp/order_status_icon/all.png",
                    'color' => null,
                    'sort_order' => 0,
                ),
                array(
                    'id'   => 1,
                    'status' => 1,
                    'code' => 'TO_PAY',
                    'name' => 'Pending',
                    'image' => "https://www.etcm.me/image/doctorapp/order_status_icon/to-pay.png",
                    'color' => '#D1D1D1',
                    'sort_order' => 1,
                ),
                array(
                    'id'   => 2,
                    'status' => 2,
                    'code' => 'TO_SHIP',
                    'name' => 'Preparing',
                    'image' => "https://www.etcm.me/image/doctorapp/order_status_icon/to-ship.png",
                    'color' => '#F38733',
                    'sort_order' => 2,
                ),
                array(
                    'id'   => 2,
                    'status' => 3,
                    'code' => 'TO_RECEIVE',
                    'name' => 'Self Collect/Shipped',
                    'image' => "https://www.etcm.me/image/doctorapp/order_status_icon/to-receive.png",
                    'color' => '#FF78BB',
                    'sort_order' => 3,
                ),
                array(
                    'id'   => 2,
                    'status' => 5,
                    'code' => 'COMPLETED',
                    'name' => 'Completed',
                    'image' => "https://www.etcm.me/image/doctorapp/order_status_icon/completed.png",
                    'color' => '#8FBB6C',
                    'sort_order' => 4,
                ),
                array(
                    'id'   => 2,
                    'status' => 7,
                    'code' => 'CANCELLED',
                    'name' => 'Cancelled',
                    'image' => "https://www.etcm.me/image/doctorapp/order_status_icon/cancelled.png",
                    'color' => '#F56B6B',
                    'sort_order' => 5,
                ),
            ];
        } else {
            $data = [
                array(
                    'id'   => 1,
                    'status' => 0,
                    'code' => 'ALL',
                    'name' => '全部',
                    'image' => "https://www.etcm.me/image/doctorapp/order_status_icon/all.png",
                    'color' => null,
                    'sort_order' => 0,
                ),
                array(
                    'id'   => 1,
                    'status' => 1,
                    'code' => 'TO_PAY',
                    'name' => '待定',
                    'image' => "https://www.etcm.me/image/doctorapp/order_status_icon/to-pay.png",
                    'color' => '#D1D1D1',
                    'sort_order' => 1,
                ),
                array(
                    'id'   => 2,
                    'status' => 1,
                    'code' => 'TO_SHIP',
                    'name' => '准备中',
                    'image' => "https://www.etcm.me/image/doctorapp/order_status_icon/to-ship.png",
                    'color' => '#F38733',
                    'sort_order' => 2,
                ),
                array(
                    'id'   => 2,
                    'status' => 3,
                    'code' => 'TO_RECEIVE',
                    'name' => '待自取/已出货',
                    'image' => "https://www.etcm.me/image/doctorapp/order_status_icon/to-receive.png",
                    'color' => '#FF78BB',
                    'sort_order' => 3,
                ),
                array(
                    'id'   => 2,
                    'status' => 5,
                    'code' => 'COMPLETED',
                    'name' => '已完成',
                    'image' => "https://www.etcm.me/image/doctorapp/order_status_icon/completed.png",
                    'color' => '#8FBB6C',
                    'sort_order' => 4,
                ),
                array(
                    'id'   => 2,
                    'status' => 7,
                    'code' => 'CANCELLED',
                    'name' => '已取消',
                    'image' => "https://www.etcm.me/image/doctorapp/order_status_icon/cancelled.png",
                    'color' => '#F56B6B',
                    'sort_order' => 5,
                ),
            ];
        }
        return $data;
    }

    private function getOrderSummary($order)
    {
        $estimated_text = $order['estimated_date'];
        if (date('Y', strtotime($estimated_text)) == '1970') {
            $estimated_text = null;
        }
        switch ($order['etcm_status_code']) {
            case 'TO_PAY':
                $title_en = 'Order Pending';
                $title_cn = '订单待定中';
                $content_en = 'Payment not complete';
                $content_cn = '未付款';
                $image = 'https://www.etcm.me/image/doctorapp/order_status_icon/to-pay-alt.png';
                break;
            case 'TO_SHIP':
                if (!empty($order['date_prepared'])) {
                    $title_en = 'Order has started prepare on';
                    $title_cn = '订单将开始准备';
                    $content_en = date_format(date_create($order['date_prepared']), "j M Y h:i A");
                    $content_cn = $content_en;
                } else {
                    $title_en = 'Order has been placed on';
                    $title_cn = '订单已付款';
                    $content_en = date_format(date_create($order['date_ordered']), "j M Y h:i A");
                    $content_cn = $content_en;
                }
                $image = 'https://www.etcm.me/image/doctorapp/order_status_icon/to-ship-alt.png';
                break;
            case 'TO_RECEIVE':
                if ($order['shipping_type'] == 'DELIVERY') {
                    $title_en = 'Order has been shipped on';
                    $title_cn = '订单运送中';
                    $content_en = date_format(date_create($order['date_shipped']), "j M Y h:i A");
                    $content_cn = $content_en;
                } else {
                    $collect_date = !empty($estimated_text) ? $estimated_text : $order['date_shipped'];
                    $title_en = 'Order ready to be collected by';
                    $title_cn = '订单将被取货于';
                    $content_en = date_format(date_create($collect_date), "j M Y h:i A");
                    $content_cn = $content_en;
                }
                $image = 'https://www.etcm.me/image/doctorapp/order_status_icon/to-receive-alt.png';
                break;
            case 'COMPLETED':
                $title_en = 'Order has completed on';
                $title_cn = '订单已完成';
                $content_en = date_format(date_create($order['date_completed']), "j M Y h:i A");
                $content_cn = $content_en;
                $image = 'https://www.etcm.me/image/doctorapp/order_status_icon/completed-alt.png';
                break;
            case 'CANCELLED':
                $title_en = 'Order Cancelled';
                $title_cn = '订单已取消';
                $content_en = 'Cancel on ' . date_format(date_create($order['date_cancelled']), "j M Y h:i A");
                $content_cn = 'Cancel on ' . date_format(date_create($order['date_cancelled']), "j M Y h:i A");
                $image = 'https://www.etcm.me/image/doctorapp/order_status_icon/cancelled-alt.png';
                break;
            default:
                $title_en = '';
                $title_cn = '';
                $content_en = '';
                $content_cn = '';
                $image = 'https://www.etcm.me/image/doctorapp/order_status_icon/all-alt.png';
        }

        return [
            "title" => ($this->session->data['language_id'] == '1') ? $title_en : $title_cn,
            "content" => ($this->session->data['language_id'] == '1') ? $content_en : $content_cn,
            "image" => $image
        ];
    }

    private function formatTelephone($telephone)
    {
        $telephone_formatted = '';
        $mobile_code = '+60';
        if (substr($telephone, 0, 3) == $mobile_code) {
            $telephone_formatted = ltrim($telephone, $mobile_code);
        } else {
            if (substr($telephone, 0, 1) == '0') {
                $telephone_formatted = ltrim($telephone, '0');
            } elseif (substr($telephone, 0, 2) == '60') {
                $telephone_formatted = ltrim($telephone, '60');
            }
        }
        return "60" . $telephone_formatted;
    }

    private function getServiceTypeName($type)
    {
        $name = str_replace('_', ' ', $type);
        $service_types = $this->getServiceTypes();
        foreach ($service_types as $service_type) {
            if ($service_type['code'] == $type) {
                $name = $service_type['name'];
                break;
            }
            if (!empty($service_type['child'])) {
                foreach ($service_type['child'] as $service_type2) {
                    if ($service_type2['code'] == $type) {
                        $name = $service_type2['name'];
                        break 2;
                    }
                }
            }
        }
        return $name;
    }

    private function getServiceTypes()
    {
        $data = [];
        $language = !empty($_SERVER['HTTP_ACCEPT_LANGUAGE']) ? $_SERVER['HTTP_ACCEPT_LANGUAGE'] : 'cn';
        if ($language == 'en') {
            $data = [
                array(
                    'id'   => 1,
                    'code' => 'DECOCTION',
                    'name' => 'Decoction',
                    'sort_order' => 0,
                    'child'  => [
                        array(
                            'id'   => 2,
                            'code' => 'DECOCTION_BY_PARTNER',
                            'name' => 'Decoction: By Service Partner',
                            'sort_order' => 0
                        ),
                        array(
                            'id'   => 3,
                            'code' => 'DECOCTION_BRING_YOUR_OWN_HERBS',
                            'name' => 'Decoction: Own Prepared Herbs',
                            'sort_order' => 0
                        ),
                    ]
                ),
                array(
                    'id'   => 4,
                    'code' => 'GRANULE',
                    'name' => 'Granule',
                    'sort_order' => 0,
                    'child'  => [],
                ),
                array(
                    'id'   => 5,
                    'code' => 'HERBS',
                    'name' => 'Herbs',
                    'sort_order' => 0,
                    'child'  => [],
                ),
                /*
                //temporary disabled
                array(
                    'id'   => 6,
                    'code' => 'OINTMENT',
                    'name' => 'Ointment',
                    'sort_order' => 0,
                    'child'  => [],
                ),
                array(
                    'id'   => 7,
                    'code' => 'POWDER',
                    'name' => 'Powder',
                    'sort_order' => 0,
                    'child'  => [],
                ),
                */
            ];
        } else {
            $data = [
                array(
                    'id'   => 1,
                    'code' => 'DECOCTION',
                    'name' => '煎剂',
                    'sort_order' => 0,
                    'child'  => [
                        array(
                            'id'   => 2,
                            'code' => 'DECOCTION_BY_PARTNER',
                            'name' => '煎煮',
                            'sort_order' => 0
                        ),
                        array(
                            'id'   => 3,
                            'code' => 'DECOCTION_BRING_YOUR_OWN_HERBS',
                            'name' => '自备草药',
                            'sort_order' => 0
                        ),
                    ],
                ),
                array(
                    'id'   => 4,
                    'code' => 'GRANULE',
                    'name' => '颗粒剂',
                    'sort_order' => 0,
                    'child'  => [],
                ),
                array(
                    'id'   => 5,
                    'code' => 'HERBS',
                    'name' => '抓药',
                    'sort_order' => 0,
                    'child'  => [],
                ),
                /*
                //temporary disabled
                array(
                    'id'   => 6,
                    'code' => 'OINTMENT',
                    'name' => '膏剂',
                    'sort_order' => 0,
                    'child'  => [],
                ),
                array(
                    'id'   => 7,
                    'code' => 'POWDER',
                    'name' => '打粉',
                    'sort_order' => 0,
                    'child'  => [],
                ),
                */
            ];
        }
        return $data;
    }

    private function connectCurl($path, $data)
    {
        $language = !empty($_SERVER['HTTP_ACCEPT_LANGUAGE']) ? $_SERVER['HTTP_ACCEPT_LANGUAGE'] : 'cn';
        if ($language == 'zh') {
            $language = 'cn';
        }

        $device_id = !empty($_SERVER['HTTP_DEVICE_ID']) ? $_SERVER['HTTP_DEVICE_ID'] : '';

        $url = HTTP_API . 'index.php?route=' . $path;
        $curl = curl_init();

        curl_setopt($curl, CURLOPT_HEADER, false);
        curl_setopt($curl, CURLINFO_HEADER_OUT, true);
        curl_setopt($curl, CURLOPT_USERAGENT, $this->request->server['HTTP_USER_AGENT']);
        curl_setopt($curl, CURLOPT_URL, $url);
        curl_setopt($curl, CURLOPT_POST, 1);
        curl_setopt($curl, CURLOPT_POSTFIELDS, htmlspecialchars_decode(json_encode($data)));
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($curl, CURLOPT_HTTPHEADER, array('Accept-Language: ' . $language, 'Device-Id: ' . $device_id, 'Content-Type: application/json'));
        curl_setopt($curl, CURLOPT_TIMEOUT, 15);

        $response = curl_exec($curl);

        $result = json_decode(htmlspecialchars_decode($response), true);

        curl_close($curl);

        // var_dump($response);
        return $result;
    }

    private function sendNotification($etcm_purchase_order_id)
    {
        $status_to_notify = ['TO_SHIP', 'COMPLETED', 'CANCELLED'];
        $this->load->model('android/common/notification');
        $so_info = $this->model_name->getSObyEtcmId($etcm_purchase_order_id);

        if (in_array($so_info['etcm_status_code'], $status_to_notify)) {
            $notification_type = 'service_partner_sale_order';
            $app_url = $this->model_android_common_notification->getNotificationDataUrl('doctorapp', $notification_type);
            $data = array("url" => str_replace(':order-id', $so_info['sale_order_id'], $app_url));

            $users = $this->model_name->getRelatedPIC($so_info['store_id']);

            if (!empty($users)) {
                foreach ($users as $user_id) {
                    $language_id = $this->model_android_common_notification->getUserLanguageId($user_id);
                    $notification = $this->generateNotification($so_info, $language_id);
                    $params = array(
                        "user_id" => $user_id,
                        "notification" => $notification,
                        "notification_type" => $notification_type,
                        "data" => $data
                    );
                    $this->load->controller('android/common/notification/push', $params);
                }
            }
        }
    }

    private function generateNotification($info, $language_id)
    {
        switch ($info['etcm_status_code']) {
            case 'TO_SHIP':
                $title_en = 'New Brewing Order!';
                $title_cn = '新的煎药订单！';

                $body_en = $info['customer_name'] . ' just placed new order';
                $body_cn = $info['customer_name'] . ' 刚刚下新订单';
                break;
            case 'COMPLETED':
                $title_en = 'Brewing Order' . (!empty($info['doc_no']) ? '[' . $info['doc_no'] . ']' : '') . ' has completed!';
                $title_cn = '煎药订单' . (!empty($info['doc_no']) ? '[' . $info['doc_no'] . ']' : '') . ' 已经完成！';

                $body_en = 'Click to see detail';
                $body_cn = '点击查看详情';
                break;
            case 'CANCELLED':
                $title_en = 'Brewing Order' . (!empty($info['doc_no']) ? '[' . $info['doc_no'] . ']' : '') . ' has been cancelled!';
                $title_cn = '煎药订单' . (!empty($info['doc_no']) ? '[' . $info['doc_no'] . ']' : '') . ' 已经取消了！';

                $body_en = 'Click to see detail';
                $body_cn = '点击查看详情';
                break;
            default:
                $title_en = 'Brewing Order' . (!empty($info['doc_no']) ? '[' . $info['doc_no'] . ']' : '') . ' has been updated!';
                $title_cn = '煎药订单' . (!empty($info['doc_no']) ? '[' . $info['doc_no'] . ']' : '') . ' 已经升级！';

                $body_en = 'Click to see detail';
                $body_cn = '点击查看详情';
        }

        return array(
            'title' => $language_id == '1' ? $title_en : $title_cn,
            'body' => $language_id == '1' ? $body_en : $body_cn,
        );
    }

    #endregion
}
