<?php

class ControllerAndroidAppointmentAvailability extends Controller
{
    private $json;
    private $model_name;

    public function index()
    {
        //load required model
        $this->load->model('android/appointment/availability');

        $this->model_name = $this->model_android_appointment_availability;

        require DIR_SYSTEM . "/library/start_api.php";
    }

    protected function get_list()
    {
        $results = [];
        $day = !empty($this->request->get['day']) ? $this->request->get['day'] : null;
        $doctor_id = !empty($this->request->get['doctor_id']) ? $this->request->get['doctor_id'] : null;

        $page = !empty($this->request->get['page']) ? $this->request->get['page'] : 1;
        $show = !empty($this->request->get['show']) ? $this->request->get['show'] : 10;

        $filters = array(
            "day"       => $day,
            "doctor_id" => $doctor_id,
            "start"     => ($page - 1) * $show,
            "limit"     => $show
        );

        // Get all doctors that have appointment availability setup 
        $doctors = $this->model_name->getDoctorAvailability($filters);

        if (!empty($doctors)) {
            foreach ($doctors as $doctor) {
                // Working Days
                $working_days = $this->model_name->getDoctorAvailabilityDays($doctor['user_id']);
                $working_days = array_column($working_days, 'day');
                $working_days_string = $this->checkContinuousDays($working_days);

                // Rest Days
                $days = ['Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday', 'Sunday'];
                $rest_days = array_diff($days, $working_days);
                $rest_days_string = implode(', ', $rest_days);

                // All working days availablities by doctor
                $schedules = $this->getDoctorAvailabilities($doctor['user_id'], $type = 1);

                // Profile Images
                $image = $doctor['image'];
                $image = $this->getDoctorPhotos($image, $doctor['name']);

                $results[] = [
                    'doctor_id'    => $doctor['user_id'],
                    'doctor'       => $doctor['name'],
                    'image'        => $image,
                    'working_days' => $working_days_string,
                    'rest_days'    => $rest_days_string,
                    'schedules'    => $schedules,
                ];
            }

            $output = $this->response->setCachedResponse(200, $results, "Get list success", "appointment");
        } else {
            $output = $this->response->setCachedResponse(404, new stdClass(), "Get list success", "appointment");
        }

        $this->api->trackResponse($output);
        return;
    }

    protected function get_detail()
    {
        $doctor_id = !empty($this->request->get['doctor_id']) ? $this->request->get['doctor_id'] : 0;

        $result = [];
        if (!empty($doctor_id)) {
            $filters = [
                "doctor_id" => $doctor_id,
            ];
            $doctors     = $this->model_name->getDoctorAvailability($filters);
            $doctor_name = !empty($doctors[0]['name']) ? $doctors[0]['name'] : '';
            // Profile Images
            $image = !empty($doctors[0]['image']) ? $doctors[0]['image'] : '';
            $image = $this->getDoctorPhotos($image, $doctor_name);

            // All working days availablities by doctor
            $schedules = $this->getDoctorAvailabilities($doctor_id, $type = 2);
            $result = [
                'doctor_id'           => $doctors[0]['user_id'],
                'doctor'              => $doctor_name,
                'image'               => $image,
                'available_services'  => $this->model_name->getDoctorServices($doctor_id),
                'schedules'           => $schedules,
            ];

            $output = $this->response->setCachedResponse(200, $result, "Get detail success", "appointment");
            $this->api->trackResponse($output);
            return;
        } else {
            $output = $this->response->setCachedResponse(404, new stdClass(), "Failed to get detail", "appointment");
            $this->api->trackResponse($output);
            return;
        }
    }

    protected function update()
    {
        $jsons = $this->json;
        if (empty($jsons)) {
            $output = $this->response->setResponse2(400, new stdClass(), 'missing json body', 'appointment');
            $this->api->trackResponse($output);
            return;
        }

        $availability_ids = [];
        foreach ($jsons['schedules'] as $json) {
            if (!empty($json['day']) && !empty($json['doctor_id'])) {
                $availability_ids[] = $this->model_name->updateAppointmentAvailability($json);
            }
        }

        $this->syncToEtcm();
        $output = $this->response->setResponse2(200, array("availability_id" => $availability_ids), 'success', 'appointment');
        $this->api->trackResponse($output);
        return;
    }

    protected function delete()
    {
        $json = $this->json;


        if (!empty($json)) {
            foreach ($json as $id) {
                $this->model_name->deleteAvailability($id);
            }

            $this->syncToEtcm();
            $output = $this->response->setResponse2(200, new stdClass(), 'success', 'appointment');
            $this->api->trackResponse($output);
            return;
        } else {
            $output = $this->response->setResponse2(400, new stdClass(), 'availability_id required', 'appointment');
            $this->api->trackResponse($output);
            return;
        }
    }

    protected function get_filters()
    {
        $days = array("Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday", "Sunday");

        $doctors = $this->model_name->getAllDoctors();

        $result = [
            "days" => $days,
            "doctors" => $doctors
        ];

        $output = $this->response->setResponse2(200, $result, 'success', 'appointment');
        $this->api->trackResponse($output);
        return;
    }


    //     _____      _            _         ______                _   _             
    //    |  __ \    (_)          | |       |  ____|              | | (_)            
    //    | |__) | __ ___   ____ _| |_ ___  | |__ _   _ _ __   ___| |_ _  ___  _ __  
    //    |  ___/ '__| \ \ / / _` | __/ _ \ |  __| | | | '_ \ / __| __| |/ _ \| '_ \ 
    //    | |   | |  | |\ V / (_| | ||  __/ | |  | |_| | | | | (__| |_| | (_) | | | |
    //    |_|   |_|  |_| \_/ \__,_|\__\___| |_|   \__,_|_| |_|\___|\__|_|\___/|_| |_|
    //   

    // Get Doctor Image
    private function getDoctorPhotos($image, $doctor_name)
    {
        $this->load->model('tool/image');
        if (is_file(DIR_IMAGE . $image)) {
            $image = $this->model_tool_image->resize($image, 120, 120);
        } else {
            $image = "https://ui-avatars.com/api/?background=random&bold=true&size=120&name=" . str_replace(' ', '+', $doctor_name);
        }

        return $image;
    }

    // Get Availablities by Doctor
    private function getDoctorAvailabilities($doctor_id, $type = 1)
    {
        $result = [];
        $filter_data = [
            "doctor_id" => $doctor_id,
        ];
        $availabilities = $this->model_name->getAvailabilities($filter_data);
        foreach ($availabilities as $av) {
            $working_hours = [];
            $availability_times = $this->model_name->getAvailabilityTimes($av['availability_id']);
            $return_times = [];
            foreach ($availability_times as $time) {
                $working_hours[] = $time['start_time'] . ' - ' . $time['end_time'];
                $return_times[] = [
                    'start_time'     => $time['start_time'],
                    'end_time'       => $time['end_time'],
                    'num_app'        => (int)$time['num_app'],
                    'enable_num_app' => (bool)$time['enable_num_app'],
                ];
            }

            $availability_services = $this->model_name->getAvailabilityServices($av['availability_id']);

            $services = array_column($availability_services, 'service_id');

            //$operations = $this->getOperationHours($working_hours);

            if ($type == 1) {
                $result[] = [
                    'availability_id' => $av['availability_id'],
                    'store_id'        => $av['store_id'],
                    'doctor_id'       => $av['doctor_id'],
                    'day'             => $av['day'],
                    'minutes'         => $av['minutes'],
                    'working_hours'   => $working_hours,
                    'services'        => $services,
                    //'operations'      => $operations,
                ];
            } else {
                $result[] = [
                    'availability_id' => $av['availability_id'],
                    'store_id'        => $av['store_id'],
                    'doctor_id'       => $av['doctor_id'],
                    'day'             => $av['day'],
                    'duration'        => $av['minutes'],
                    'services'        => $services,
                    'schedule'        => $return_times,
                    'allowed_edit'    => true
                ];
            }
        }
        return $result;
    }

    // Get the working day string
    // If continuosly will use `-`, e,g Monday - Friday
    // If are not continously, will use `,` Monday, Wednesday
    private function checkContinuousDays(array $workingDays)
    {
        $days = ['Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday', 'Sunday'];
        $workingDays = array_intersect($days, $workingDays);
        usort($workingDays, function ($a, $b) use ($days) {
            return array_search($a, $days) > array_search($b, $days) ? 1 : -1;
        });
        $result = "";
        $start = 0;
        $end = 0;
        for ($i = 0; $i < count($workingDays); $i++) {
            if ($i == 0) {
                $start = array_search($workingDays[$i], $days);
                $end = $start;
                continue;
            }
            if ($days[array_search($workingDays[$i], $days) - 1] != $workingDays[$i - 1]) {
                if ($start == $end) {
                    $result .= $workingDays[$i - 1] . ", ";
                } else {
                    $result .= $workingDays[$start] . " - " . $workingDays[$i - 1] . ", ";
                }
                $start = array_search($workingDays[$i], $days);
                $end = $start;
            } else {
                $end = array_search($workingDays[$i], $days);
            }
        }
        if ($start == $end) {
            $result .= $workingDays[count($workingDays) - 1];
        } else {
            $result .= $workingDays[$start] . " - " . $workingDays[$end];
        }
        return $result;
    }

    // Get Operation Hours by current time
    private function getOperationHours(array $workingHours)
    {
        $result = [
            'status' => '',
            'closes' => '',
            'reopen' => '',
            'operation' => ''
        ];
        date_default_timezone_set("Asia/Kuala_Lumpur");
        $now = new DateTime();
        $currentTime = $now->format('h:i A');
        $isOpen = false;
        $closes = '';
        $reopen = '';
        foreach ($workingHours as $workingHour) {
            $hours = explode(' - ', $workingHour);
            if ($currentTime >= $hours[0] && $currentTime < $hours[1]) {
                $isOpen = true;
                $closes = $hours[1];
                break;
            } elseif ($currentTime < $hours[0]) {
                $reopen = $hours[0];
                break;
            }
        }
        if ($isOpen) {
            $result['status'] = 'Open';
            $result['closes'] = $closes;
            $result['operation'] = 'Open ⋅ Closes ' . $closes;
        } else {
            $result['status'] = 'Closed';
            $result['reopen'] = $reopen;
            $result['operation'] = 'Closed ⋅ Reopens ' . $reopen;
        }
        return $result;
    }

    private function syncToEtcm()
    {

        $db_availabilities = $this->db->query("SELECT * FROM " . DB_PREFIX . "appointment_availability")->rows;
        $doctor_availabilities = [];
        foreach ($db_availabilities as $availability) {
            $db_times = $this->db->query("SELECT * FROM " . DB_PREFIX . "appointment_availability_time WHERE availability_id = '" . (int)$availability['availability_id'] . "'")->rows;
            $db_services = $this->db->query("SELECT * FROM " . DB_PREFIX . "availability_to_service WHERE availability_id = '" . (int)$availability['availability_id'] . "'")->rows;
            $doctor_availabilities[] = [
                'availability_id'   => $availability['availability_id'],
                'doctor_id'         => $availability['doctor_id'],
                'store_id'          => $availability['store_id'],
                'minutes'           => $availability['minutes'],
                'day'               => $availability['day'],
                'times'             => $db_times,
                'services'          => $db_services,
            ];
        }
        $post_data = [
            'store_url' => HTTPS_SERVER,
            'doctor_availabilities' => $doctor_availabilities
        ];

        $path = 'etcm/syncing/syncing/update_doctor_availabilities';
        $result = $this->connectCurl($path, $post_data);
        return $result;
    }

    private function connectCurl($path, $data)
    {
        $language = !empty($_SERVER['HTTP_ACCEPT_LANGUAGE']) ? $_SERVER['HTTP_ACCEPT_LANGUAGE'] : 'cn';
        if ($language == 'zh') {
            $language = 'cn';
        }

        $device_id = !empty($_SERVER['HTTP_DEVICE_ID']) ? $_SERVER['HTTP_DEVICE_ID'] : '';

        $url = 'https://www.etcm.me/index.php?route=' . $path;
        $curl = curl_init();

        curl_setopt($curl, CURLOPT_HEADER, false);
        curl_setopt($curl, CURLINFO_HEADER_OUT, true);
        curl_setopt($curl, CURLOPT_USERAGENT, $this->request->server['HTTP_USER_AGENT']);
        curl_setopt($curl, CURLOPT_URL, $url);
        curl_setopt($curl, CURLOPT_POST, 1);
        curl_setopt($curl, CURLOPT_POSTFIELDS, htmlspecialchars_decode(json_encode($data)));
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($curl, CURLOPT_HTTPHEADER, array('Accept-Language: ' . $language, 'Device-Id: ' . $device_id, 'Content-Type: application/json'));
        curl_setopt($curl, CURLOPT_TIMEOUT, 15);

        $response = curl_exec($curl);

        $result = json_decode(htmlspecialchars_decode($response), true);

        curl_close($curl);

        return $result;
    }
}
