<?php

class ControllerSaleExcelImport extends Controller {

    private $error = [];

    public function index() {
        $this->load->language('sale/order');
        $this->load->model('sale/order');
        $this->load->model('sale/shipping_module');
        $this->load->language('sale/shipping_module');
        $this->load->language('sale/excel_import');

        $this->document->setTitle($this->language->get('heading_title'));

        $this->load->model('sale/excel_import');
        $this->load->model('sale/customer');
        $this->load->model('setting/store');
        $this->load->model('localisation/country');

        $this->load->library('tax');
        $this->tax = new Tax($this->registry);

        // = new SMS($this->config->get('config_sms'));

        if ($this->request->server['REQUEST_METHOD'] == 'POST' && $this->validateForm()) {
            if (isset($this->request->post['transactions'])) {
                $notify_customer_when_shipped = $this->request->post['notify_customer_when_shipped'];

                $store_id = $this->session->data['store_id'];
                $store_info = $this->model_setting_store->getStore($store_id);

                $store_name = !empty($store_info) ? $store_info['name'] : $this->config->get('config_name');
                $store_url = !empty($store_info) ? $store_info['url'] : HTTP_CATALOG;

                foreach ($this->request->post['transactions'] as $transaction) {
                    $address = [];

                    $transaction_json = base64_decode($transaction);
                    $transaction_arr = json_decode($transaction_json, true);

                    $customer_group_row = $this->model_sale_excel_import->get_setting_by_key('customer_group');
                    $customer_group_id = !empty($customer_group_row) ? $customer_group_row['value'] : 0;

                    $order_status_row = $this->model_sale_excel_import->get_setting_by_key('order_status');
                    $order_status_id = !empty($order_status_row) ? $order_status_row['value'] : 0;

                    $order_status_with_courier_row = $this->model_sale_excel_import->get_setting_by_key('order_status_with_courier');
                    $order_status_with_courier_id = !empty($order_status_with_courier_row) ? $order_status_with_courier_row['value'] : 0;

                    $customer_info = $this->model_sale_excel_import->get_customer_by_telephone($transaction_arr['telephone']);

                    if (empty($customer_info)) {
                        if (!empty($transaction_arr['customer_address'])) {
                            $address[] = [
                                'firstname' => $transaction_arr['customer_name'],
                                'lastname' => '',
                                'company' => '',
                                'address_1' => $transaction_arr['customer_address']['address_1'],
                                'address_2' => '',
                                'address_3' => '',
                                'city' => $transaction_arr['customer_address']['city'],
                                'postcode' => $transaction_arr['customer_address']['postcode'],
                                'zone_id' => $transaction_arr['customer_address']['zone_id'],
                                'country_id' => $transaction_arr['customer_address']['country_id'],
                                'default' => 1
                            ];
                        } else {
                            $address = [];
                        }

                        $insert_customer_data = [
                            'firstname' => $transaction_arr['customer_name'],
                            'lastname' => '',
                            'customer_ic' => '',
                            'customer_card' => '',
                            'dob' => '',
                            'gender' => '0',
                            'customer_group_id' => $customer_group_id,
                            'email' => '',
                            'telephone' => $transaction_arr['telephone'],
                            'telephone2' => '',
                            'home' => '',
                            'fax' => '',
                            'newsletter' => '0',
                            'status' => '1',
                            'password' => '',
                            'approved' => '1',
                            'safe' => '1'
                        ];

                        if (!empty($address)) {
                            $insert_customer_data['address'] = $address;
                        }

                        $customer_id = $this->model_sale_customer->addCustomer($insert_customer_data);
                    } else {
                        $customer_id = $customer_info['customer_id'];
                    }

                    $customer_info = $this->model_sale_excel_import->get_customer($customer_id);
                    $address_id = !empty($customer_info) ? $customer_info['address_id'] : 0;

                    $address_info = $this->model_sale_excel_import->get_address($address_id);

                    if (!empty($customer_info)) {
                        $order_products = [];
                        $order_totals = [];

                        date_default_timezone_set($this->config->get('config_time_zone'));

                        $total_tax = 0;

                        foreach ($transaction_arr['product_purchased'] as $product) {
                            $product_info = $this->model_sale_excel_import->get_product($product['product_id']);

                            if (!empty($product_info)) {
                                $product_total_tax = $this->tax->getTax($product['unit_price'], $product_info['tax_class_id']) * $product['quantity'];

                                $order_products[] = [
                                    'product_id' => $product['product_id'],
                                    'name' => $product['product_name'],
                                    'model' => $product['model'],
                                    'quantity' => $product['quantity'],
                                    'tax_class_id' => $product_info['tax_class_id'],
                                    'price' => $product['unit_price'],
                                    'net_amount' => $product['unit_price'],
                                    'total_net_amount' => $product['total_price'],
                                    'total' => $product['total_price'],
                                    'tax' => $this->tax->getTax($product['unit_price'], $product_info['tax_class_id']),
                                    'total_tax' => $product_total_tax
                                ];

                                $total_tax += $product_total_tax;
                            }
                        }

                        $order_totals[] = [
                            'code' => 'sub_total',
                            'title' => 'Total Sales (Excl Tax)',
                            'value' => $transaction_arr['system_total'],
                            'sort_order' => '1'
                        ];

                        $order_totals[] = [
                            'code' => 'shipping',
                            'title' => 'Delivery Charge',
                            'value' => 0,
                            'sort_order' => '3'
                        ];

                        $order_totals[] = [
                            'code' => 'tax',
                            'title' => 'Tax',
                            'value' => $total_tax,
                            'sort_order' => '5'
                        ];

                        $order_totals[] = [
                            'code' => 'rounding',
                            'title' => 'Rounding',
                            'value' => 0,
                            'sort_order' => '8'
                        ];

                        $order_totals[] = [
                            'code' => 'total',
                            'title' => 'Total Sales (Incl Tax)',
                            'value' => $transaction_arr['system_total'] + $total_tax,
                            'sort_order' => '9'
                        ];

                        $insert_order_data = [
                            'store_id' => $store_id,
                            'store_name' => $store_name,
                            'store_url' => $store_url,
                            'customer_id' => $customer_id,
                            'customer_group_id' => $customer_info['customer_group_id'],
                            'firstname' => $customer_info['firstname'],
                            'email' => $customer_info['email'],
                            'telephone' => $customer_info['telephone'],
                            'payment_firstname' => $customer_info['firstname'],
                            'payment_address_1' => !empty($address_info) ? $address_info['address_1'] : '',
                            'payment_address_2' => !empty($address_info) ? $address_info['address_2'] : '',
                            'payment_city' => !empty($address_info) ? $address_info['city'] : '',
                            'payment_postcode' => !empty($address_info) ? $address_info['postcode'] : '',
                            'payment_country' => !empty($transaction_arr['customer_address']) ? $transaction_arr['customer_address']['country_name'] : '',
                            'payment_country_id' => !empty($address_info) ? $address_info['country_id'] : 0,
                            'payment_zone_id' => !empty($address_info) ? $address_info['zone_id'] : 0,
                            'payment_zone' => !empty($transaction_arr['customer_address']) ? $transaction_arr['customer_address']['zone_name'] : '',
                            'payment_method' => $transaction_arr['payment_method'],
                            'payment_code' => 'bank_transfer',
                            'shipping_address_id' => !empty($address_info) ? $address_info['address_id'] : 0,
                            'shipping_firstname' => $customer_info['firstname'],
                            'shipping_address_1' => !empty($address_info) ? $address_info['address_1'] : '',
                            'shipping_address_2' => !empty($address_info) ? $address_info['address_2'] : '',
                            'shipping_city' => !empty($address_info) ? $address_info['city'] : '',
                            'shipping_postcode' => !empty($address_info) ? $address_info['postcode'] : '',
                            'shipping_country' => !empty($transaction_arr['customer_address']) ? $transaction_arr['customer_address']['country_name'] : '',
                            'shipping_country_id' => !empty($address_info) ? $address_info['country_id'] : 0,
                            'shipping_zone_id' => !empty($address_info) ? $address_info['zone_id'] : 0,
                            'shipping_zone' => !empty($transaction_arr['customer_address']) ? $transaction_arr['customer_address']['zone_name'] : '',
                            'shipping_method' => 'Free Delivery',
                            'shipping_code' => 'category_product_based.category_product_based_0',
                            'comment' => '',
                            'total' => $transaction_arr['system_total'] + $total_tax,
                            'shipping_total' => 0,
                            'order_status_id' => $order_status_id,
                            'language_id' => $this->config->get('config_language_id'),
                            'currency_id' => $this->currency->getId($this->config->get('config_currency')),
                            'currency_code' => $this->config->get('config_currency'),
                            'currency_value' => $this->currency->getValue($this->config->get('config_currency')),
                            'ip' => '',
                            'user_agent' => '',
                            'date_ordered' => date('Y-m-d', strtotime($transaction_arr['transaction_date'])),
                            'order_products' => $order_products,
                            'order_totals' => $order_totals
                        ];

                        $tracking_number = !empty($transaction_arr['tracking_number']) ? $transaction_arr['tracking_number'] : '';
                        $courier_service = !empty($transaction_arr['courier_service']) ? $transaction_arr['courier_service'] : '';
                        $courier_comment = !empty($transaction_arr['courier_comment']) ? $transaction_arr['courier_comment'] : '';

                        $country_id = $this->config->get('config_country_id');
                        $country_info = $this->model_localisation_country->getCountry($country_id);
                        $mobile_code = $country_info['mobile_code'];

                        $order_tracking_number_sms_settings = $this->config->get('config_sms_customer_order_tracking_number');

                        // if the order is already imported
                        if ($transaction_arr['import_order_id'] > 0) {
                            $order_id = $transaction_arr['import_order_id'];

                            $order_info = $this->model_sale_shipping_module->getOrder($order_id);

                            if (!empty($tracking_number) && !empty($courier_service)) {
                                // check order history with shipped for this order is exists or not
                                $shipped_order_history_row = $this->model_sale_excel_import->get_order_history($order_id, $order_status_with_courier_id);

                                $tracking_url = $this->model_sale_shipping_module->getTrackingURL($courier_service);

                                $find_tracking_num = ['{trackingnumber}'];

                                $replace_tracking_num = [
                                    'trackingnumber' => $tracking_number
                                ];

                                $tracking_link = str_replace($find_tracking_num, $replace_tracking_num, $tracking_url);

                                if (empty($shipped_order_history_row)) {
                                    // check notify customer or not
                                    if ($notify_customer_when_shipped > 0) {
                                        // send sms to the customer for the tracking number 
                                        if (!empty($customer_info['telephone'])) {
                                            $customer_telephone = $customer_info['telephone'];

                                            if (substr($customer_telephone, 0, 2) == '+6') {
                                                $customer_telephone = ltrim($customer_telephone, '+6');
                                            }

                                            if (substr($customer_telephone, 0, 1) == '6') {
                                                $customer_telephone = ltrim($customer_telephone, '6');
                                            }

                                            if ($order_tracking_number_sms_settings['status'] == '1') {
                                                $find_sms_template = ['{order_id}', '{courier}', '{shipping_tracking}'];

                                                $replace_sms_template = [
                                                    'order_id' => $order_id,
                                                    'courier' => $courier_service,
                                                    'shipping_tracking' => $tracking_number
                                                ];

                                                $sms_message = str_replace($find_sms_template, $replace_sms_template, $order_tracking_number_sms_settings['template']);

                                                $send_sms->sendTo($mobile_code, $customer_telephone);

                                                if (preg_match('/[^A-Za-z0-9\s\.\,]/', $sms_message)) {
                                                    $send_sms->changeContentType();
                                                    $send_sms->convertUnicode($sms_message);
                                                } else {
                                                    $send_sms->setMessage($sms_message);
                                                }

                                                $send_sms->send();
                                            }
                                        }

                                        if (!empty($customer_info['email'])) {
                                            $data['title'] = sprintf($this->language->get('text_subject'), $order_info['store_name'], $order_info['order_id']);

                                            $data['text_greeting'] = sprintf($this->language->get('text_greeting'), $order_info['store_name']);

                                            $order_products = $this->model_sale_order->getOrderProducts($order_info['order_id']);

                                            $data['logo'] = $order_info['store_url'] . 'image/' . $this->config->get('config_logo');

                                            $data['store_name'] = $order_info['store_name'];
                                            $data['store_url'] = $order_info['store_url'];
                                            $data['customer_id'] = $order_info['customer_id'];
                                            $data['link'] = $tracking_link;

                                            $data['order_id'] = $order_info['order_id'];
                                            $data['date_added'] = date($this->language->get('date_format_short'), strtotime($order_info['date_added']));
                                            $data['payment_method'] = $order_info['payment_method'];
                                            $data['shipping_method'] = $order_info['shipping_method'];
                                            $data['email'] = $order_info['email'];
                                            $data['telephone'] = $order_info['telephone'];
                                            $data['ip'] = $order_info['ip'];

                                            date_default_timezone_set("Asia/Kuala_Lumpur");
                                            $data['now_time'] = date("d/M/y h:i:s A");

                                            $order_status_query = $this->db->query("SELECT * FROM " . DB_PREFIX . "order_status WHERE order_status_id = '" . (int) $order_status_id . "' AND language_id = '" . (int) $order_info['language_id'] . "'");

                                            if ($order_status_query->num_rows) {
                                                $data['order_status'] = $order_status_query->row['name'];
                                            } else {
                                                $data['order_status'] = '';
                                            }

                                            if ($tracking_link) {
                                                $data['comment'] = nl2br($tracking_link);
                                            } else {
                                                $data['comment'] = '';
                                            }

                                            if ($order_info['payment_address_format']) {
                                                $format = $order_info['payment_address_format'];
                                            } else {
                                                $format = '{firstname} {lastname}' . "\n" . '{company}' . "\n" . '{address_1}' . "\n" . '{address_2}' . "\n" . '{city} {postcode}' . "\n" . '{zone}' . "\n" . '{country}';
                                            }

                                            $find = array(
                                                '{firstname}',
                                                '{lastname}',
                                                '{company}',
                                                '{address_1}',
                                                '{address_2}',
                                                '{city}',
                                                '{postcode}',
                                                '{zone}',
                                                '{zone_code}',
                                                '{country}'
                                            );

                                            $replace = array(
                                                'firstname' => $order_info['payment_firstname'],
                                                'lastname' => $order_info['payment_lastname'],
                                                'company' => $order_info['payment_company'],
                                                'address_1' => $order_info['payment_address_1'],
                                                'address_2' => $order_info['payment_address_2'],
                                                'city' => $order_info['payment_city'],
                                                'postcode' => $order_info['payment_postcode'],
                                                'zone' => $order_info['payment_zone'],
                                                'zone_code' => $order_info['payment_zone_code'],
                                                'country' => $order_info['payment_country']
                                            );

                                            $data['payment_address'] = str_replace(array("\r\n", "\r", "\n"), '<br />', preg_replace(array("/\s\s+/", "/\r\r+/", "/\n\n+/"), '<br />', trim(str_replace($find, $replace, $format))));

                                            if ($order_info['shipping_address_format']) {
                                                $format = $order_info['shipping_address_format'];
                                            } else {
                                                $format = '{firstname} {lastname}' . "\n" . '{company}' . "\n" . '{address_1}' . "\n" . '{address_2}' . "\n" . '{city} {postcode}' . "\n" . '{zone}' . "\n" . '{country}';
                                            }

                                            $find = array(
                                                '{firstname}',
                                                '{lastname}',
                                                '{company}',
                                                '{address_1}',
                                                '{address_2}',
                                                '{city}',
                                                '{postcode}',
                                                '{zone}',
                                                '{zone_code}',
                                                '{country}'
                                            );

                                            $replace = array(
                                                'firstname' => $order_info['shipping_firstname'],
                                                'lastname' => $order_info['shipping_lastname'],
                                                'company' => $order_info['shipping_company'],
                                                'address_1' => $order_info['shipping_address_1'],
                                                'address_2' => $order_info['shipping_address_2'],
                                                'city' => $order_info['shipping_city'],
                                                'postcode' => $order_info['shipping_postcode'],
                                                'zone' => $order_info['shipping_zone'],
                                                'zone_code' => $order_info['shipping_zone_code'],
                                                'country' => $order_info['shipping_country']
                                            );

                                            $data['shipping_address'] = str_replace(array("\r\n", "\r", "\n"), '<br />', preg_replace(array("/\s\s+/", "/\r\r+/", "/\n\n+/"), '<br />', trim(str_replace($find, $replace, $format))));

                                            $this->load->model('tool/upload');

                                            // Products
                                            $data['products'] = array();

                                            foreach ($order_products as $order_product) {
                                                $option_data = array();

                                                $order_options = $this->model_sale_order->getOrderOptions($order_info['order_id'], $order_product['order_product_id']);

                                                foreach ($order_options as $order_option) {
                                                    if ($order_option['type'] != 'file') {
                                                        $value = $order_option['value'];
                                                    } else {
                                                        $upload_info = $this->model_tool_upload->getUploadByCode($order_option['value']);

                                                        if ($upload_info) {
                                                            $value = $upload_info['name'];
                                                        } else {
                                                            $value = '';
                                                        }
                                                    }

                                                    $option_data[] = array(
                                                        'name' => $order_option['name'],
                                                        'value' => (utf8_strlen($value) > 20 ? utf8_substr($value, 0, 20) . '..' : $value)
                                                    );
                                                }

                                                $data['products'][] = array(
                                                    'name' => $order_product['name'],
                                                    'model' => $order_product['model'],
                                                    'option' => $option_data,
                                                    'quantity' => $order_product['quantity'],
                                                    'price' => $this->currency->format($order_product['price'] + ($this->config->get('config_tax') ? $order_product['tax'] : 0), $order_info['currency_code'], $order_info['currency_value']),
                                                    'total' => $this->currency->format($order_product['total'] + ($this->config->get('config_tax') ? ($order_product['tax'] * $order_product['quantity']) : 0), $order_info['currency_code'], $order_info['currency_value'])
                                                );
                                            }

                                            // Vouchers
                                            $data['vouchers'] = array();

                                            $order_vouchers = $this->model_sale_order->getOrderVouchers($order_info['order_id']);

                                            foreach ($order_vouchers as $order_voucher) {
                                                $data['vouchers'][] = array(
                                                    'description' => $order_voucher['description'],
                                                    'amount' => $this->currency->format($order_voucher['amount'], $order_info['currency_code'], $order_info['currency_value']),
                                                );
                                            }

                                            // Order Totals
                                            $data['totals'] = array();

                                            $order_totals = $this->model_sale_order->getOrderTotals($order_info['order_id']);

                                            foreach ($order_totals as $order_total) {
                                                $data['totals'][] = array(
                                                    'title' => $order_total['title'],
                                                    'text' => $this->currency->format($order_total['value'], $order_info['currency_code'], $order_info['currency_value']),
                                                );
                                            }

                                            $this->load->model('setting/setting');

                                            $from = $this->model_setting_setting->getSettingValue('config_email', $order_info['store_id']);

                                            if (!$from) {
                                                $from = $this->config->get('config_email');
                                            }

                                            $mail = new Mail($this->config->get('config_mail_engine'));
                                            $mail->parameter = $this->config->get('config_mail_parameter');
                                            $mail->smtp_hostname = $this->config->get('config_mail_smtp_hostname');
                                            $mail->smtp_username = $this->config->get('config_mail_smtp_username');
                                            $mail->smtp_password = html_entity_decode($this->config->get('config_mail_smtp_password'), ENT_QUOTES, 'UTF-8');
                                            $mail->smtp_port = $this->config->get('config_mail_smtp_port');
                                            $mail->smtp_timeout = $this->config->get('config_mail_smtp_timeout');
                                            $mail->setTo($order_info['email']);
                                            $mail->setFrom($from);
                                            $mail->setSender(html_entity_decode($order_info['store_name'], ENT_QUOTES, 'UTF-8'));
                                            $mail->setSubject(html_entity_decode(sprintf($order_info['store_name'], ENT_QUOTES, 'UTF-8')));
                                            $mail->setHtml($this->load->view('sale/notify_customer_email', $data));
                                            $mail->send();
                                        }
                                    }

                                    // add shipped order history
                                    $this->model_sale_excel_import->add_order_history($order_id, $order_status_with_courier_id, $notify_customer_when_shipped, $courier_comment);

                                    // update order status of the order
                                    $this->model_sale_excel_import->update_order_status($order_id, $order_status_with_courier_id);

                                    // update the carrier
                                    $courier_info = !empty($courier_service) ? $this->model_sale_excel_import->get_courier_data_by_name($courier_service) : [];
                                    $carrier_id = !empty($courier_info) ? $courier_info['carrier_id'] : 0;

                                    $this->model_sale_excel_import->update_carrier($order_id, $carrier_id, $tracking_number);
                                }
                            }
                        } else {
                            // if the order havent import 
                            $order_id = $this->model_sale_excel_import->add_order($insert_order_data);
                            $order_info = $this->model_sale_shipping_module->getOrder($order_id);
                            $customer_info = $this->model_sale_shipping_module->getCustomerInfo($order_info['customer_id']);

                            if (!empty($tracking_number) && !empty($courier_service)) {

                                $tracking_url = $this->model_sale_shipping_module->getTrackingURL($courier_service);

                                $find_tracking_num = ['{trackingnumber}'];

                                $replace_tracking_num = [
                                    'trackingnumber' => $tracking_number
                                ];

                                $tracking_link = str_replace($find_tracking_num, $replace_tracking_num, $tracking_url);

                                // set the order status to shipped
                                $this->model_sale_excel_import->update_order_status($order_id, $order_status_with_courier_id);

                                // update the carrier
                                $courier_info = !empty($courier_service) ? $this->model_sale_excel_import->get_courier_data_by_name($courier_service) : [];
                                $carrier_id = !empty($courier_info) ? $courier_info['carrier_id'] : 0;

                                $this->model_sale_excel_import->update_carrier($order_id, $carrier_id, $tracking_number);

                                $order_info = $this->model_sale_order->getOrder($order_id);
                                // check notify customer or not
                                if ($notify_customer_when_shipped > 0) {
                                    // send sms to the customer for the tracking number 
                                    if (!empty($customer_info['telephone'])) {
                                        $customer_telephone = $customer_info['telephone'];

                                        if (substr($customer_telephone, 0, 2) == '+6') {
                                            $customer_telephone = ltrim($customer_telephone, '+6');
                                        }

                                        if (substr($customer_telephone, 0, 1) == '6') {
                                            $customer_telephone = ltrim($customer_telephone, '6');
                                        }

                                        if ($order_tracking_number_sms_settings['status'] == '1') {
                                            $find_sms_template = ['{order_id}', '{courier}', '{shipping_tracking}'];

                                            $replace_sms_template = [
                                                'order_id' => $order_id,
                                                'courier' => $courier_service,
                                                'shipping_tracking' => $tracking_number
                                            ];

                                            $sms_message = str_replace($find_sms_template, $replace_sms_template, $order_tracking_number_sms_settings['template']);

                                            $send_sms->sendTo($mobile_code, $customer_telephone);

                                            if (preg_match('/[^A-Za-z0-9\s\.\,]/', $sms_message)) {
                                                $send_sms->changeContentType();
                                                $send_sms->convertUnicode($sms_message);
                                            } else {
                                                $send_sms->setMessage($sms_message);
                                            }

                                            $send_sms->send();
                                        }
                                    }

                                    if (!empty($customer_info['email'])) {
                                        $data['title'] = sprintf($this->language->get('text_subject'), $order_info['store_name'], $order_info['order_id']);

                                        $data['text_greeting'] = sprintf($this->language->get('text_greeting'), $order_info['store_name']);

                                        $order_products = $this->model_sale_order->getOrderProducts($order_info['order_id']);

                                        $data['logo'] = $order_info['store_url'] . 'image/' . $this->config->get('config_logo');

                                        $data['store_name'] = $order_info['store_name'];
                                        $data['store_url'] = $order_info['store_url'];
                                        $data['customer_id'] = $order_info['customer_id'];
                                        $data['link'] = $tracking_link;

                                        $data['order_id'] = $order_info['order_id'];
                                        $data['date_added'] = date($this->language->get('date_format_short'), strtotime($order_info['date_added']));
                                        $data['payment_method'] = $order_info['payment_method'];
                                        $data['shipping_method'] = $order_info['shipping_method'];
                                        $data['email'] = $order_info['email'];
                                        $data['telephone'] = $order_info['telephone'];
                                        $data['ip'] = $order_info['ip'];

                                        date_default_timezone_set("Asia/Kuala_Lumpur");
                                        $data['now_time'] = date("d/M/y h:i:s A");

                                        $order_status_query = $this->db->query("SELECT * FROM " . DB_PREFIX . "order_status WHERE order_status_id = '" . (int) $order_status_id . "' AND language_id = '" . (int) $order_info['language_id'] . "'");

                                        if ($order_status_query->num_rows) {
                                            $data['order_status'] = $order_status_query->row['name'];
                                        } else {
                                            $data['order_status'] = '';
                                        }

                                        if ($tracking_link) {
                                            $data['comment'] = nl2br($tracking_link);
                                        } else {
                                            $data['comment'] = '';
                                        }

                                        if ($order_info['payment_address_format']) {
                                            $format = $order_info['payment_address_format'];
                                        } else {
                                            $format = '{firstname} {lastname}' . "\n" . '{company}' . "\n" . '{address_1}' . "\n" . '{address_2}' . "\n" . '{city} {postcode}' . "\n" . '{zone}' . "\n" . '{country}';
                                        }

                                        $find = array(
                                            '{firstname}',
                                            '{lastname}',
                                            '{company}',
                                            '{address_1}',
                                            '{address_2}',
                                            '{city}',
                                            '{postcode}',
                                            '{zone}',
                                            '{zone_code}',
                                            '{country}'
                                        );

                                        $replace = array(
                                            'firstname' => $order_info['payment_firstname'],
                                            'lastname' => $order_info['payment_lastname'],
                                            'company' => $order_info['payment_company'],
                                            'address_1' => $order_info['payment_address_1'],
                                            'address_2' => $order_info['payment_address_2'],
                                            'city' => $order_info['payment_city'],
                                            'postcode' => $order_info['payment_postcode'],
                                            'zone' => $order_info['payment_zone'],
                                            'zone_code' => $order_info['payment_zone_code'],
                                            'country' => $order_info['payment_country']
                                        );

                                        $data['payment_address'] = str_replace(array("\r\n", "\r", "\n"), '<br />', preg_replace(array("/\s\s+/", "/\r\r+/", "/\n\n+/"), '<br />', trim(str_replace($find, $replace, $format))));

                                        if ($order_info['shipping_address_format']) {
                                            $format = $order_info['shipping_address_format'];
                                        } else {
                                            $format = '{firstname} {lastname}' . "\n" . '{company}' . "\n" . '{address_1}' . "\n" . '{address_2}' . "\n" . '{city} {postcode}' . "\n" . '{zone}' . "\n" . '{country}';
                                        }

                                        $find = array(
                                            '{firstname}',
                                            '{lastname}',
                                            '{company}',
                                            '{address_1}',
                                            '{address_2}',
                                            '{city}',
                                            '{postcode}',
                                            '{zone}',
                                            '{zone_code}',
                                            '{country}'
                                        );

                                        $replace = array(
                                            'firstname' => $order_info['shipping_firstname'],
                                            'lastname' => $order_info['shipping_lastname'],
                                            'company' => $order_info['shipping_company'],
                                            'address_1' => $order_info['shipping_address_1'],
                                            'address_2' => $order_info['shipping_address_2'],
                                            'city' => $order_info['shipping_city'],
                                            'postcode' => $order_info['shipping_postcode'],
                                            'zone' => $order_info['shipping_zone'],
                                            'zone_code' => $order_info['shipping_zone_code'],
                                            'country' => $order_info['shipping_country']
                                        );

                                        $data['shipping_address'] = str_replace(array("\r\n", "\r", "\n"), '<br />', preg_replace(array("/\s\s+/", "/\r\r+/", "/\n\n+/"), '<br />', trim(str_replace($find, $replace, $format))));

                                        $this->load->model('tool/upload');

                                        // Products
                                        $data['products'] = array();

                                        foreach ($order_products as $order_product) {
                                            $option_data = array();

                                            $order_options = $this->model_sale_order->getOrderOptions($order_info['order_id'], $order_product['order_product_id']);

                                            foreach ($order_options as $order_option) {
                                                if ($order_option['type'] != 'file') {
                                                    $value = $order_option['value'];
                                                } else {
                                                    $upload_info = $this->model_tool_upload->getUploadByCode($order_option['value']);

                                                    if ($upload_info) {
                                                        $value = $upload_info['name'];
                                                    } else {
                                                        $value = '';
                                                    }
                                                }

                                                $option_data[] = array(
                                                    'name' => $order_option['name'],
                                                    'value' => (utf8_strlen($value) > 20 ? utf8_substr($value, 0, 20) . '..' : $value)
                                                );
                                            }

                                            $data['products'][] = array(
                                                'name' => $order_product['name'],
                                                'model' => $order_product['model'],
                                                'option' => $option_data,
                                                'quantity' => $order_product['quantity'],
                                                'price' => $this->currency->format($order_product['price'] + ($this->config->get('config_tax') ? $order_product['tax'] : 0), $order_info['currency_code'], $order_info['currency_value']),
                                                'total' => $this->currency->format($order_product['total'] + ($this->config->get('config_tax') ? ($order_product['tax'] * $order_product['quantity']) : 0), $order_info['currency_code'], $order_info['currency_value'])
                                            );
                                        }

                                        // Vouchers
                                        $data['vouchers'] = array();

                                        $order_vouchers = $this->model_sale_order->getOrderVouchers($order_info['order_id']);

                                        foreach ($order_vouchers as $order_voucher) {
                                            $data['vouchers'][] = array(
                                                'description' => $order_voucher['description'],
                                                'amount' => $this->currency->format($order_voucher['amount'], $order_info['currency_code'], $order_info['currency_value']),
                                            );
                                        }

                                        // Order Totals
                                        $data['totals'] = array();

                                        $order_totals = $this->model_sale_order->getOrderTotals($order_info['order_id']);

                                        foreach ($order_totals as $order_total) {
                                            $data['totals'][] = array(
                                                'title' => $order_total['title'],
                                                'text' => $this->currency->format($order_total['value'], $order_info['currency_code'], $order_info['currency_value']),
                                            );
                                        }

                                        $this->load->model('setting/setting');

                                        $from = $this->model_setting_setting->getSettingValue('config_email', $order_info['store_id']);

                                        if (!$from) {
                                            $from = $this->config->get('config_email');
                                        }

                                        $mail = new Mail($this->config->get('config_mail_engine'));
                                        $mail->parameter = $this->config->get('config_mail_parameter');
                                        $mail->smtp_hostname = $this->config->get('config_mail_smtp_hostname');
                                        $mail->smtp_username = $this->config->get('config_mail_smtp_username');
                                        $mail->smtp_password = html_entity_decode($this->config->get('config_mail_smtp_password'), ENT_QUOTES, 'UTF-8');
                                        $mail->smtp_port = $this->config->get('config_mail_smtp_port');
                                        $mail->smtp_timeout = $this->config->get('config_mail_smtp_timeout');
                                        $mail->setTo($order_info['email']);
                                        $mail->setFrom($from);
                                        $mail->setSender(html_entity_decode($order_info['store_name'], ENT_QUOTES, 'UTF-8'));
                                        $mail->setSubject(html_entity_decode(sprintf($order_info['store_name']), ENT_QUOTES, 'UTF-8'));
                                        $mail->setHtml($this->load->view('sale/notify_customer_email', $data));
                                        $mail->send();
                                    }
                                }

                                // add order history of shipped
                                $this->model_sale_excel_import->add_order_history($order_id, $order_status_with_courier_id, $notify_customer_when_shipped, $courier_comment);
                            }
                        }
                    }
                }
            }

            $this->session->data['success'] = $this->language->get('text_success_import');

            $this->response->redirect($this->url->link('sale/excel_import', 'user_token=' . $this->session->data['user_token'], true));
        }

        $this->getForm();
    }

    public function import_excel() {
        $json = [];
        $full_return_transactions = [];
        $return_transactions = [];

        $this->load->model('sale/excel_import');
        $this->load->model('localisation/country');

        $excel_filename = $this->request->files['excel_file']['name'];

        $ext = pathinfo($excel_filename, PATHINFO_EXTENSION);

        if (strtolower($ext) != 'xlsx') {
            $json = [
                'success' => false
            ];
        } else {
            $this->load->library('PHPExcel');
            $this->excel = new PHPExcel();

            $obj_reader = PHPExcel_IOFactory::createReader('Excel2007');
            $obj_php_excel = $obj_reader->load($this->request->files['excel_file']['tmp_name']);

            $sheet = $obj_php_excel->getActiveSheet();
            $highest_row = $sheet->getHighestRow();
            $highest_column = $sheet->getHighestColumn();

            $sheets = [];

            for ($row = 1; $row <= $highest_row; $row++) {
                $sheets[$row] = $sheet->rangeToArray('A' . $row . ':' . $highest_column . $highest_row, null, true, false)[0];
            }

            $transactions_imported = [];
            $headers = array_shift($sheets);
            $header = array_filter($headers);

            foreach ($sheets as $d => $dt) {
                if (is_array($dt)) {
                    $transactions_imported[$d] = [];
                    foreach ($dt as $rd => $row) {
                        if (in_array($rd, array_keys($header))) {
                            $transactions_imported[$d][$header[$rd]] = $row;
                        }
                    }
                }
            }

            if (!empty($transactions_imported)) {
                foreach ($transactions_imported as $transaction) {
                    $got_order_date = array_key_exists('Order Date', $transaction) ? true : false;
                    $got_telephone = array_key_exists('Contact Number', $transaction) ? true : false;
                    $got_customer_name = array_key_exists('Customer Name', $transaction) ? true : false;
                    $got_payment_method = array_key_exists('Payment Method', $transaction) ? true : false;
                    $got_product_code = array_key_exists('Product Code', $transaction) ? true : false;
                    $got_address_1 = array_key_exists('Delivery Address 1', $transaction) ? true : false;
                    $got_postcode = array_key_exists('Postcode', $transaction) ? true : false;
                    $got_city = array_key_exists('Shipping City', $transaction) ? true : false;
                    $got_state = array_key_exists('Shipping State', $transaction) ? true : false;
                    $got_amount = array_key_exists('Amount', $transaction) ? true : false;
                    $got_courier_service = array_key_exists('Courier Service', $transaction) ? true : false;
                    $got_tracking_number = array_key_exists('Tracking Number', $transaction) ? true : false;

                    if ($got_order_date && $got_telephone && $got_customer_name && $got_payment_method && $got_product_code && $got_address_1 && $got_postcode && $got_city && $got_state && $got_amount && $got_courier_service && $got_tracking_number) {
                        $product_purchased_arr = [];
                        $grandtotal_price = 0;

                        $find_telephone = ['-', ' '];
                        $replace_telephone = ['dash' => '', 'space' => ''];

                        $excel_telephone = str_replace($find_telephone, $replace_telephone, $transaction['Contact Number']);
                        $excel_telephone_arr = explode('/', $excel_telephone);
                        $telephone = $excel_telephone_arr[0];

                        $customer_info = $this->model_sale_excel_import->get_customer_by_telephone($telephone);
                        $db_customer_name = !empty($customer_info) ? $customer_info['firstname'] . ' ' . $customer_info['lastname'] : '-';
                        $customer_url = !empty($customer_info['customer_id']) ? $this->url->link('customer/customer/edit', 'user_token=' . $this->session->data['user_token'] . '&customer_id=' . $customer_info['customer_id'], true) : '';
                        $db_customer_id = !empty($customer_info['customer_id']) ? $customer_info['customer_id'] : 0;

                        $transaction_date = date('Y-m-d', PHPExcel_Shared_Date::ExcelToPHP($transaction['Order Date']));

                        $product_purchased = $transaction['Product Code'];
                        $product_purchased_split = explode('/', $product_purchased);

                        foreach ($product_purchased_split as $product) {
                            $total_price = 0;

                            $product_split = explode('*', $product);

                            $product_code = $product_split[0];
                            $purchase_quantity = isset($product_split[1]) ? $product_split[1] : 1;

                            $product_info = $this->model_sale_excel_import->get_product_by_product_code($product_code);

                            if (!empty($product_info)) {
                                $total_price = $purchase_quantity * $product_info['price'];

                                $product_purchased_arr[] = [
                                    'product_id' => $product_info['product_id'],
                                    'product_name' => $product_info['product_name'],
                                    'model' => $product_info['model'],
                                    'product_code' => $product_code,
                                    'unit_price' => $product_info['price'],
                                    'quantity' => (int) $purchase_quantity,
                                    'total_price' => $total_price
                                ];

                                $grandtotal_price += $total_price;
                            }
                        }

                        $zone_name = $transaction['Shipping State'];
                        $zone_info = $this->model_sale_excel_import->get_zone_by_name($zone_name);

                        $country_info = !empty($zone_info) ? $this->model_localisation_country->getCountry($zone_info['country_id']) : [];

                        $address = [
                            'address_1' => ucwords(strtolower($transaction['Delivery Address 1'])),
                            'postcode' => $transaction['Postcode'],
                            'city' => ucwords(strtolower($transaction['Shipping City'])),
                            'zone_id' => !empty($zone_info) ? $zone_info['zone_id'] : 0,
                            'zone_name' => !empty($zone_info) ? $zone_info['name'] : '',
                            'country_id' => !empty($zone_info) ? $zone_info['country_id'] : 0,
                            'country_name' => !empty($country_info) ? $country_info['name'] : ''
                        ];

                        $same_total_status = ((string)$transaction['Amount'] == (string)$grandtotal_price) ? true : false;

                        // check order imported or not
                        $order_details = !empty($db_customer_id) ? $this->model_sale_excel_import->get_excel_import_order($transaction_date, $db_customer_id) : [];
                        $order_id = !empty($order_details['order_id']) ? $order_details['order_id'] : 0;

                        $imported_order = ($order_id > 0) ? true : false;
                        $order_url = ($order_id > 0) ? $this->url->link('sale/order/info', 'user_token=' . $this->session->data['user_token'] . '&order_id=' . $order_id, true) : '';

                        // manage courier service and tracking number
                        $courier_service = !empty($transaction['Courier Service']) ? $transaction['Courier Service'] : '';
                        $tracking_number = !empty($transaction['Tracking Number']) ? $transaction['Tracking Number'] : '';

                        $courier_data = !empty($courier_service) ? $this->model_sale_excel_import->get_courier_data_by_name($courier_service) : [];
                        $courier_comment = !empty($courier_data) ? $courier_data['name'] . ' ' . $tracking_number . ' ' . str_replace('{tracking_number}', $tracking_number, $courier_data['tracking_url']) : '';

                        $return_transactions[] = [
                            'transaction_date' => $transaction_date,
                            'db_customer_id' => (int) $db_customer_id,
                            'db_customer_name' => $db_customer_name,
                            'customer_url' => $customer_url,
                            'customer_name' => ucwords(strtolower($transaction['Customer Name'])),
                            'customer_address' => !empty($address['address_1']) ? $address : [],
                            'telephone' => $telephone,
                            'total' => $transaction['Amount'],
                            'total_formatted' => $this->currency->format($transaction['Amount'], $this->config->get('config_currency')),
                            'system_total' => $grandtotal_price,
                            'system_total_formatted' => $this->currency->format($grandtotal_price, $this->config->get('config_currency')),
                            'same_total_status' => $same_total_status,
                            'product_purchased' => $product_purchased_arr,
                            'payment_method' => $transaction['Payment Method'],
                            'import_order_id' => (int) $order_id,
                            'imported_order' => $imported_order,
                            'order_url' => $order_url,
                            'courier_service' => $courier_service,
                            'tracking_number' => $tracking_number,
                            'courier_comment' => $courier_comment
                        ];
                    }
                }
            }

            if (!empty($return_transactions)) {
                $transaction_date_column = array_column($return_transactions, 'transaction_date');
                $transaction_date_column = array_map('strtotime', $transaction_date_column);

                array_multisort($return_transactions, SORT_NUMERIC, $transaction_date_column, SORT_ASC);

                foreach ($return_transactions as $transaction) {
                    $full_return_transactions[] = [
                        'data' => $transaction,
                        'encrypted_data' => base64_encode(json_encode($transaction))
                    ];
                }

                $json = array(
                    'success' => true,
                    'total_transaction' => count($full_return_transactions),
                    'respond' => $full_return_transactions
                );
            } else {
                $json = [
                    'success' => false
                ];
            }
        }

        $this->response->addHeader('Content-Type: application/json');
        $this->response->setOutput(json_encode($json));
    }

    public function settings() {
        $this->load->language('sale/excel_import');

        $this->document->setTitle($this->language->get('heading_title'));

        $this->load->model('sale/excel_import');

        if (($this->request->server['REQUEST_METHOD'] == 'POST') && $this->validateSettingsForm()) {
            $this->model_sale_excel_import->save_settings($this->request->post);

            $this->session->data['success'] = $this->language->get('text_success_import_settings');

            $this->response->redirect($this->url->link('sale/excel_import', 'user_token=' . $this->session->data['user_token'], true));
        }

        $this->getSettingsForm();
    }

    protected function getForm() {

        $url = '';

        $data['breadcrumbs'] = array();

        $data['breadcrumbs'][] = array(
            'text' => $this->language->get('text_home'),
            'href' => $this->url->link('common/dashboard', 'user_token=' . $this->session->data['user_token'], true)
        );

        $data['breadcrumbs'][] = array(
            'text' => $this->language->get('heading_title'),
            'href' => $this->url->link('sale/excel_import', 'user_token=' . $this->session->data['user_token'] . $url, true)
        );

        $data['error_text_warning'] = $this->language->get('error_warning');

        if (isset($this->session->data['success'])) {
            $data['success'] = $this->session->data['success'];

            unset($this->session->data['success']);
        } else {
            $data['success'] = '';
        }

        if (isset($this->error['warning'])) {
            $data['error_warning'] = $this->error['warning'];
        } else {
            $data['error_warning'] = '';
        }

        if (isset($this->request->post['transactions'])) {
            $data['transactions'] = $this->request->post['transactions'];
        } else {
            $data['transactions'] = [];
        }

        if (isset($this->request->post['notify_customer_when_shipped'])) {
            $data['notify_customer_when_shipped'] = $this->request->post['notify_customer_when_shipped'];
        } else {
            $data['notify_customer_when_shipped'] = '0';
        }

        $data['total_transaction'] = 0;

        $data['user_token'] = $this->session->data['user_token'];

        $data['action'] = $this->url->link('sale/excel_import', 'user_token=' . $this->session->data['user_token'], true);
        $data['import_excel_action'] = $this->url->link('sale/excel_import/import_excel', 'user_token=' . $this->session->data['user_token'], true);
        $data['excel_import_settings'] = $this->url->link('sale/excel_import/settings', 'user_token=' . $this->session->data['user_token'], true);

        $data['sample_excel'] = HTTPS_SERVER . 'sample/excel_sample_transaction_format.xlsx';

        $data['header'] = $this->load->controller('common/header');
        $data['column_left'] = $this->load->controller('common/column_left');
        $data['footer'] = $this->load->controller('common/footer');

        $this->response->setOutput($this->load->view('sale/excel_import_form', $data));
    }

    protected function getSettingsForm() {
        $import_settings = $this->get_settings();

        $url = '';

        $data['breadcrumbs'] = array();

        $data['breadcrumbs'][] = array(
            'text' => $this->language->get('text_home'),
            'href' => $this->url->link('common/dashboard', 'user_token=' . $this->session->data['user_token'], true)
        );

        $data['breadcrumbs'][] = array(
            'text' => $this->language->get('heading_title'),
            'href' => $this->url->link('sale/excel_import', 'user_token=' . $this->session->data['user_token'] . $url, true)
        );

        $data['action'] = $this->url->link('sale/excel_import/settings', 'user_token=' . $this->session->data['user_token'], true);
        $data['cancel'] = $this->url->link('sale/excel_import', 'user_token=' . $this->session->data['user_token'], true);

        if (isset($this->error['warning'])) {
            $data['error_warning'] = $this->error['warning'];
        } else {
            $data['error_warning'] = '';
        }

        if (isset($this->error['customer_group'])) {
            $data['error_customer_group'] = $this->error['customer_group'];
        } else {
            $data['error_customer_group'] = '';
        }

        if (isset($this->error['order_status'])) {
            $data['error_order_status'] = $this->error['order_status'];
        } else {
            $data['error_order_status'] = '';
        }

        if (isset($this->error['order_status_with_courier'])) {
            $data['error_order_status_with_courier'] = $this->error['order_status_with_courier'];
        } else {
            $data['error_order_status_with_courier'] = '';
        }

        if (isset($this->request->post['customer_group'])) {
            $data['customer_group'] = $this->request->post['customer_group'];
        } elseif (!empty($import_settings)) {
            $data['customer_group'] = $import_settings['customer_group'];
        } else {
            $data['customer_group'] = '';
        }

        if (isset($this->request->post['order_status'])) {
            $data['order_status'] = $this->request->post['order_status'];
        } elseif (!empty($import_settings)) {
            $data['order_status'] = $import_settings['order_status'];
        } else {
            $data['order_status'] = '';
        }

        if (isset($this->request->post['order_status_with_courier'])) {
            $data['order_status_with_courier'] = $this->request->post['order_status_with_courier'];
        } elseif (!empty($import_settings)) {
            $data['order_status_with_courier'] = $import_settings['order_status_with_courier'];
        } else {
            $data['order_status_with_courier'] = '';
        }

        $this->load->model('sale/customer_group');
        $this->load->model('localisation/order_status');

        $data['customer_groups'] = $this->model_sale_customer_group->getCustomerGroups();
        $data['order_statuses'] = $this->model_localisation_order_status->getOrderStatuses();

        $data['header'] = $this->load->controller('common/header');
        $data['column_left'] = $this->load->controller('common/column_left');
        $data['footer'] = $this->load->controller('common/footer');

        $this->response->setOutput($this->load->view('sale/excel_settings_form', $data));
    }

    protected function validateForm() {
        if (!$this->user->hasPermission('modify', 'sale/excel_import')) {
            $this->error['warning'] = $this->language->get('error_permission');
        }

        if (empty($this->request->post['transactions'])) {
            $this->error['warning'] = $this->language->get('error_import_transaction');
        }

        if ($this->error && !isset($this->error['warning'])) {
            $this->error['warning'] = $this->language->get('error_warning');
        }

        return !$this->error;
    }

    protected function validateSettingsForm() {
        if (!$this->user->hasPermission('modify', 'sale/excel_import')) {
            $this->error['warning'] = $this->language->get('error_permission');
        }

        if (empty($this->request->post['customer_group'])) {
            $this->error['customer_group'] = $this->language->get('error_customer_group');
        }

        if (empty($this->request->post['order_status'])) {
            $this->error['order_status'] = $this->language->get('error_order_status');
        }

        if (empty($this->request->post['order_status_with_courier'])) {
            $this->error['order_status_with_courier'] = $this->language->get('error_order_status_with_courier');
        }

        if ($this->error && !isset($this->error['warning'])) {
            $this->error['warning'] = $this->language->get('error_warning');
        }

        return !$this->error;
    }

    private function get_settings() {
        $setting_results = [];

        $this->load->model('sale/excel_import');

        $settings = $this->model_sale_excel_import->get_settings();

        if (!empty($settings)) {
            foreach ($settings as $setting) {
                if ($setting['serialized'] == '1') {
                    $setting_results[$setting['setting_key']] = unserialize($setting['value']);
                } else {
                    $setting_results[$setting['setting_key']] = $setting['value'];
                }
            }
        }

        return $setting_results;
    }

}
